(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (factory((global.vueFeatherIcons = {})));
}(this, (function (exports) { 'use strict';

    /**
     * Make a map and return a function for checking if a key
     * is in that map.
     * IMPORTANT: all calls of this function must be prefixed with
     * \/\*#\_\_PURE\_\_\*\/
     * So that rollup can tree-shake them if necessary.
     */
    function makeMap(str, expectsLowerCase) {
        const map = Object.create(null);
        const list = str.split(',');
        for (let i = 0; i < list.length; i++) {
            map[list[i]] = true;
        }
        return expectsLowerCase ? val => !!map[val.toLowerCase()] : val => !!map[val];
    }

    const GLOBALS_WHITE_LISTED = 'Infinity,undefined,NaN,isFinite,isNaN,parseFloat,parseInt,decodeURI,' +
        'decodeURIComponent,encodeURI,encodeURIComponent,Math,Number,Date,Array,' +
        'Object,Boolean,String,RegExp,Map,Set,JSON,Intl';
    const isGloballyWhitelisted = /*#__PURE__*/ makeMap(GLOBALS_WHITE_LISTED);

    /**
     * On the client we only need to offer special cases for boolean attributes that
     * have different names from their corresponding dom properties:
     * - itemscope -> N/A
     * - allowfullscreen -> allowFullscreen
     * - formnovalidate -> formNoValidate
     * - ismap -> isMap
     * - nomodule -> noModule
     * - novalidate -> noValidate
     * - readonly -> readOnly
     */
    const specialBooleanAttrs = `itemscope,allowfullscreen,formnovalidate,ismap,nomodule,novalidate,readonly`;
    const isSpecialBooleanAttr = /*#__PURE__*/ makeMap(specialBooleanAttrs);
    /**
     * The full list is needed during SSR to produce the correct initial markup.
     */
    const isBooleanAttr = /*#__PURE__*/ makeMap(specialBooleanAttrs +
        `,async,autofocus,autoplay,controls,default,defer,disabled,hidden,` +
        `loop,open,required,reversed,scoped,seamless,` +
        `checked,muted,multiple,selected`);
    /**
     * CSS properties that accept plain numbers
     */
    const isNoUnitNumericStyleProp = /*#__PURE__*/ makeMap(`animation-iteration-count,border-image-outset,border-image-slice,` +
        `border-image-width,box-flex,box-flex-group,box-ordinal-group,column-count,` +
        `columns,flex,flex-grow,flex-positive,flex-shrink,flex-negative,flex-order,` +
        `grid-row,grid-row-end,grid-row-span,grid-row-start,grid-column,` +
        `grid-column-end,grid-column-span,grid-column-start,font-weight,line-clamp,` +
        `line-height,opacity,order,orphans,tab-size,widows,z-index,zoom,` +
        // SVG
        `fill-opacity,flood-opacity,stop-opacity,stroke-dasharray,stroke-dashoffset,` +
        `stroke-miterlimit,stroke-opacity,stroke-width`);
    /**
     * Known attributes, this is used for stringification of runtime static nodes
     * so that we don't stringify bindings that cannot be set from HTML.
     * Don't also forget to allow `data-*` and `aria-*`!
     * Generated from https://developer.mozilla.org/en-US/docs/Web/HTML/Attributes
     */
    const isKnownAttr = /*#__PURE__*/ makeMap(`accept,accept-charset,accesskey,action,align,allow,alt,async,` +
        `autocapitalize,autocomplete,autofocus,autoplay,background,bgcolor,` +
        `border,buffered,capture,challenge,charset,checked,cite,class,code,` +
        `codebase,color,cols,colspan,content,contenteditable,contextmenu,controls,` +
        `coords,crossorigin,csp,data,datetime,decoding,default,defer,dir,dirname,` +
        `disabled,download,draggable,dropzone,enctype,enterkeyhint,for,form,` +
        `formaction,formenctype,formmethod,formnovalidate,formtarget,headers,` +
        `height,hidden,high,href,hreflang,http-equiv,icon,id,importance,integrity,` +
        `ismap,itemprop,keytype,kind,label,lang,language,loading,list,loop,low,` +
        `manifest,max,maxlength,minlength,media,min,multiple,muted,name,novalidate,` +
        `open,optimum,pattern,ping,placeholder,poster,preload,radiogroup,readonly,` +
        `referrerpolicy,rel,required,reversed,rows,rowspan,sandbox,scope,scoped,` +
        `selected,shape,size,sizes,slot,span,spellcheck,src,srcdoc,srclang,srcset,` +
        `start,step,style,summary,tabindex,target,title,translate,type,usemap,` +
        `value,width,wrap`);

    function normalizeStyle(value) {
        if (isArray(value)) {
            const res = {};
            for (let i = 0; i < value.length; i++) {
                const item = value[i];
                const normalized = normalizeStyle(isString(item) ? parseStringStyle(item) : item);
                if (normalized) {
                    for (const key in normalized) {
                        res[key] = normalized[key];
                    }
                }
            }
            return res;
        }
        else if (isObject(value)) {
            return value;
        }
    }
    const listDelimiterRE = /;(?![^(]*\))/g;
    const propertyDelimiterRE = /:(.+)/;
    function parseStringStyle(cssText) {
        const ret = {};
        cssText.split(listDelimiterRE).forEach(item => {
            if (item) {
                const tmp = item.split(propertyDelimiterRE);
                tmp.length > 1 && (ret[tmp[0].trim()] = tmp[1].trim());
            }
        });
        return ret;
    }
    function normalizeClass(value) {
        let res = '';
        if (isString(value)) {
            res = value;
        }
        else if (isArray(value)) {
            for (let i = 0; i < value.length; i++) {
                res += normalizeClass(value[i]) + ' ';
            }
        }
        else if (isObject(value)) {
            for (const name in value) {
                if (value[name]) {
                    res += name + ' ';
                }
            }
        }
        return res.trim();
    }

    // These tag configs are shared between compiler-dom and runtime-dom, so they
    // https://developer.mozilla.org/en-US/docs/Web/HTML/Element
    const HTML_TAGS = 'html,body,base,head,link,meta,style,title,address,article,aside,footer,' +
        'header,h1,h2,h3,h4,h5,h6,hgroup,nav,section,div,dd,dl,dt,figcaption,' +
        'figure,picture,hr,img,li,main,ol,p,pre,ul,a,b,abbr,bdi,bdo,br,cite,code,' +
        'data,dfn,em,i,kbd,mark,q,rp,rt,rtc,ruby,s,samp,small,span,strong,sub,sup,' +
        'time,u,var,wbr,area,audio,map,track,video,embed,object,param,source,' +
        'canvas,script,noscript,del,ins,caption,col,colgroup,table,thead,tbody,td,' +
        'th,tr,button,datalist,fieldset,form,input,label,legend,meter,optgroup,' +
        'option,output,progress,select,textarea,details,dialog,menu,menuitem,' +
        'summary,content,element,shadow,template,blockquote,iframe,tfoot';
    // https://developer.mozilla.org/en-US/docs/Web/SVG/Element
    const SVG_TAGS = 'svg,animate,animateMotion,animateTransform,circle,clipPath,color-profile,' +
        'defs,desc,discard,ellipse,feBlend,feColorMatrix,feComponentTransfer,' +
        'feComposite,feConvolveMatrix,feDiffuseLighting,feDisplacementMap,' +
        'feDistanceLight,feDropShadow,feFlood,feFuncA,feFuncB,feFuncG,feFuncR,' +
        'feGaussianBlur,feImage,feMerge,feMergeNode,feMorphology,feOffset,' +
        'fePointLight,feSpecularLighting,feSpotLight,feTile,feTurbulence,filter,' +
        'foreignObject,g,hatch,hatchpath,image,line,linearGradient,marker,mask,' +
        'mesh,meshgradient,meshpatch,meshrow,metadata,mpath,path,pattern,' +
        'polygon,polyline,radialGradient,rect,set,solidcolor,stop,switch,symbol,' +
        'text,textPath,title,tspan,unknown,use,view';
    const VOID_TAGS = 'area,base,br,col,embed,hr,img,input,link,meta,param,source,track,wbr';
    const isHTMLTag = /*#__PURE__*/ makeMap(HTML_TAGS);
    const isSVGTag = /*#__PURE__*/ makeMap(SVG_TAGS);
    const isVoidTag = /*#__PURE__*/ makeMap(VOID_TAGS);

    const EMPTY_OBJ = Object.freeze({});
    const EMPTY_ARR = [];
    const NOOP = () => { };
    const onRE = /^on[^a-z]/;
    const isOn = (key) => onRE.test(key);
    const extend = Object.assign;
    const remove = (arr, el) => {
        const i = arr.indexOf(el);
        if (i > -1) {
            arr.splice(i, 1);
        }
    };
    const hasOwnProperty = Object.prototype.hasOwnProperty;
    const hasOwn = (val, key) => hasOwnProperty.call(val, key);
    const isArray = Array.isArray;
    const isFunction = (val) => typeof val === 'function';
    const isString = (val) => typeof val === 'string';
    const isSymbol = (val) => typeof val === 'symbol';
    const isObject = (val) => val !== null && typeof val === 'object';
    const isPromise = (val) => {
        return isObject(val) && isFunction(val.then) && isFunction(val.catch);
    };
    const objectToString = Object.prototype.toString;
    const toTypeString = (value) => objectToString.call(value);
    const toRawType = (value) => {
        return toTypeString(value).slice(8, -1);
    };
    const isReservedProp = /*#__PURE__*/ makeMap('key,ref,' +
        'onVnodeBeforeMount,onVnodeMounted,' +
        'onVnodeBeforeUpdate,onVnodeUpdated,' +
        'onVnodeBeforeUnmount,onVnodeUnmounted');
    const cacheStringFunction = (fn) => {
        const cache = Object.create(null);
        return ((str) => {
            const hit = cache[str];
            return hit || (cache[str] = fn(str));
        });
    };
    const camelizeRE = /-(\w)/g;
    const camelize = cacheStringFunction((str) => {
        return str.replace(camelizeRE, (_, c) => (c ? c.toUpperCase() : ''));
    });
    const hyphenateRE = /\B([A-Z])/g;
    const hyphenate = cacheStringFunction((str) => {
        return str.replace(hyphenateRE, '-$1').toLowerCase();
    });
    const capitalize = cacheStringFunction((str) => {
        return str.charAt(0).toUpperCase() + str.slice(1);
    });
    // compare whether a value has changed, accounting for NaN.
    const hasChanged = (value, oldValue) => value !== oldValue && (value === value || oldValue === oldValue);
    const def = (obj, key, value) => {
        Object.defineProperty(obj, key, {
            configurable: true,
            enumerable: false,
            value
        });
    };
    const toNumber = (val) => {
        const n = parseFloat(val);
        return isNaN(n) ? val : n;
    };

    const targetMap = new WeakMap();
    const effectStack = [];
    let activeEffect;
    const ITERATE_KEY = Symbol('iterate');
    const MAP_KEY_ITERATE_KEY = Symbol('Map key iterate');
    function isEffect(fn) {
        return fn && fn._isEffect === true;
    }
    function effect(fn, options = EMPTY_OBJ) {
        if (isEffect(fn)) {
            fn = fn.raw;
        }
        const effect = createReactiveEffect(fn, options);
        if (!options.lazy) {
            effect();
        }
        return effect;
    }
    function stop(effect) {
        if (effect.active) {
            cleanup(effect);
            if (effect.options.onStop) {
                effect.options.onStop();
            }
            effect.active = false;
        }
    }
    let uid = 0;
    function createReactiveEffect(fn, options) {
        const effect = function reactiveEffect(...args) {
            if (!effect.active) {
                return options.scheduler ? undefined : fn(...args);
            }
            if (!effectStack.includes(effect)) {
                cleanup(effect);
                try {
                    enableTracking();
                    effectStack.push(effect);
                    activeEffect = effect;
                    return fn(...args);
                }
                finally {
                    effectStack.pop();
                    resetTracking();
                    activeEffect = effectStack[effectStack.length - 1];
                }
            }
        };
        effect.id = uid++;
        effect._isEffect = true;
        effect.active = true;
        effect.raw = fn;
        effect.deps = [];
        effect.options = options;
        return effect;
    }
    function cleanup(effect) {
        const { deps } = effect;
        if (deps.length) {
            for (let i = 0; i < deps.length; i++) {
                deps[i].delete(effect);
            }
            deps.length = 0;
        }
    }
    let shouldTrack = true;
    const trackStack = [];
    function pauseTracking() {
        trackStack.push(shouldTrack);
        shouldTrack = false;
    }
    function enableTracking() {
        trackStack.push(shouldTrack);
        shouldTrack = true;
    }
    function resetTracking() {
        const last = trackStack.pop();
        shouldTrack = last === undefined ? true : last;
    }
    function track(target, type, key) {
        if (!shouldTrack || activeEffect === undefined) {
            return;
        }
        let depsMap = targetMap.get(target);
        if (!depsMap) {
            targetMap.set(target, (depsMap = new Map()));
        }
        let dep = depsMap.get(key);
        if (!dep) {
            depsMap.set(key, (dep = new Set()));
        }
        if (!dep.has(activeEffect)) {
            dep.add(activeEffect);
            activeEffect.deps.push(dep);
            if (activeEffect.options.onTrack) {
                activeEffect.options.onTrack({
                    effect: activeEffect,
                    target,
                    type,
                    key
                });
            }
        }
    }
    function trigger(target, type, key, newValue, oldValue, oldTarget) {
        const depsMap = targetMap.get(target);
        if (!depsMap) {
            // never been tracked
            return;
        }
        const effects = new Set();
        const add = (effectsToAdd) => {
            if (effectsToAdd) {
                effectsToAdd.forEach(effect => {
                    if (effect !== activeEffect || !shouldTrack) {
                        effects.add(effect);
                    }
                });
            }
        };
        if (type === "clear" /* CLEAR */) {
            // collection being cleared
            // trigger all effects for target
            depsMap.forEach(add);
        }
        else if (key === 'length' && isArray(target)) {
            depsMap.forEach((dep, key) => {
                if (key === 'length' || key >= newValue) {
                    add(dep);
                }
            });
        }
        else {
            // schedule runs for SET | ADD | DELETE
            if (key !== void 0) {
                add(depsMap.get(key));
            }
            // also run for iteration key on ADD | DELETE | Map.SET
            const isAddOrDelete = type === "add" /* ADD */ ||
                (type === "delete" /* DELETE */ && !isArray(target));
            if (isAddOrDelete ||
                (type === "set" /* SET */ && target instanceof Map)) {
                add(depsMap.get(isArray(target) ? 'length' : ITERATE_KEY));
            }
            if (isAddOrDelete && target instanceof Map) {
                add(depsMap.get(MAP_KEY_ITERATE_KEY));
            }
        }
        const run = (effect) => {
            if (effect.options.onTrigger) {
                effect.options.onTrigger({
                    effect,
                    target,
                    key,
                    type,
                    newValue,
                    oldValue,
                    oldTarget
                });
            }
            if (effect.options.scheduler) {
                effect.options.scheduler(effect);
            }
            else {
                effect();
            }
        };
        effects.forEach(run);
    }

    const builtInSymbols = new Set(Object.getOwnPropertyNames(Symbol)
        .map(key => Symbol[key])
        .filter(isSymbol));
    const get = /*#__PURE__*/ createGetter();
    const shallowGet = /*#__PURE__*/ createGetter(false, true);
    const readonlyGet = /*#__PURE__*/ createGetter(true);
    const shallowReadonlyGet = /*#__PURE__*/ createGetter(true, true);
    const arrayInstrumentations = {};
    ['includes', 'indexOf', 'lastIndexOf'].forEach(key => {
        arrayInstrumentations[key] = function (...args) {
            const arr = toRaw(this);
            for (let i = 0, l = this.length; i < l; i++) {
                track(arr, "get" /* GET */, i + '');
            }
            // we run the method using the original args first (which may be reactive)
            const res = arr[key](...args);
            if (res === -1 || res === false) {
                // if that didn't work, run it again using raw values.
                return arr[key](...args.map(toRaw));
            }
            else {
                return res;
            }
        };
    });
    function createGetter(isReadonly = false, shallow = false) {
        return function get(target, key, receiver) {
            if (key === "__v_isReactive" /* IS_REACTIVE */) {
                return !isReadonly;
            }
            else if (key === "__v_isReadonly" /* IS_READONLY */) {
                return isReadonly;
            }
            else if (key === "__v_raw" /* RAW */ &&
                receiver ===
                    (isReadonly
                        ? target["__v_readonly" /* READONLY */]
                        : target["__v_reactive" /* REACTIVE */])) {
                return target;
            }
            const targetIsArray = isArray(target);
            if (targetIsArray && hasOwn(arrayInstrumentations, key)) {
                return Reflect.get(arrayInstrumentations, key, receiver);
            }
            const res = Reflect.get(target, key, receiver);
            if (isSymbol(key)
                ? builtInSymbols.has(key)
                : key === `__proto__` || key === `__v_isRef`) {
                return res;
            }
            if (!isReadonly) {
                track(target, "get" /* GET */, key);
            }
            if (shallow) {
                return res;
            }
            if (isRef(res)) {
                // ref unwrapping, only for Objects, not for Arrays.
                return targetIsArray ? res : res.value;
            }
            if (isObject(res)) {
                // Convert returned value into a proxy as well. we do the isObject check
                // here to avoid invalid value warning. Also need to lazy access readonly
                // and reactive here to avoid circular dependency.
                return isReadonly ? readonly(res) : reactive(res);
            }
            return res;
        };
    }
    const set = /*#__PURE__*/ createSetter();
    const shallowSet = /*#__PURE__*/ createSetter(true);
    function createSetter(shallow = false) {
        return function set(target, key, value, receiver) {
            const oldValue = target[key];
            if (!shallow) {
                value = toRaw(value);
                if (!isArray(target) && isRef(oldValue) && !isRef(value)) {
                    oldValue.value = value;
                    return true;
                }
            }
            const hadKey = hasOwn(target, key);
            const result = Reflect.set(target, key, value, receiver);
            // don't trigger if target is something up in the prototype chain of original
            if (target === toRaw(receiver)) {
                if (!hadKey) {
                    trigger(target, "add" /* ADD */, key, value);
                }
                else if (hasChanged(value, oldValue)) {
                    trigger(target, "set" /* SET */, key, value, oldValue);
                }
            }
            return result;
        };
    }
    function deleteProperty(target, key) {
        const hadKey = hasOwn(target, key);
        const oldValue = target[key];
        const result = Reflect.deleteProperty(target, key);
        if (result && hadKey) {
            trigger(target, "delete" /* DELETE */, key, undefined, oldValue);
        }
        return result;
    }
    function has(target, key) {
        const result = Reflect.has(target, key);
        track(target, "has" /* HAS */, key);
        return result;
    }
    function ownKeys(target) {
        track(target, "iterate" /* ITERATE */, ITERATE_KEY);
        return Reflect.ownKeys(target);
    }
    const mutableHandlers = {
        get,
        set,
        deleteProperty,
        has,
        ownKeys
    };
    const readonlyHandlers = {
        get: readonlyGet,
        has,
        ownKeys,
        set(target, key) {
            {
                console.warn(`Set operation on key "${String(key)}" failed: target is readonly.`, target);
            }
            return true;
        },
        deleteProperty(target, key) {
            {
                console.warn(`Delete operation on key "${String(key)}" failed: target is readonly.`, target);
            }
            return true;
        }
    };
    const shallowReactiveHandlers = extend({}, mutableHandlers, {
        get: shallowGet,
        set: shallowSet
    });
    // Props handlers are special in the sense that it should not unwrap top-level
    // refs (in order to allow refs to be explicitly passed down), but should
    // retain the reactivity of the normal readonly object.
    const shallowReadonlyHandlers = extend({}, readonlyHandlers, {
        get: shallowReadonlyGet
    });

    const toReactive = (value) => isObject(value) ? reactive(value) : value;
    const toReadonly = (value) => isObject(value) ? readonly(value) : value;
    const toShallow = (value) => value;
    const getProto = (v) => Reflect.getPrototypeOf(v);
    function get$1(target, key, wrap) {
        target = toRaw(target);
        const rawKey = toRaw(key);
        if (key !== rawKey) {
            track(target, "get" /* GET */, key);
        }
        track(target, "get" /* GET */, rawKey);
        const { has, get } = getProto(target);
        if (has.call(target, key)) {
            return wrap(get.call(target, key));
        }
        else if (has.call(target, rawKey)) {
            return wrap(get.call(target, rawKey));
        }
    }
    function has$1(key) {
        const target = toRaw(this);
        const rawKey = toRaw(key);
        if (key !== rawKey) {
            track(target, "has" /* HAS */, key);
        }
        track(target, "has" /* HAS */, rawKey);
        const has = getProto(target).has;
        return has.call(target, key) || has.call(target, rawKey);
    }
    function size(target) {
        target = toRaw(target);
        track(target, "iterate" /* ITERATE */, ITERATE_KEY);
        return Reflect.get(getProto(target), 'size', target);
    }
    function add(value) {
        value = toRaw(value);
        const target = toRaw(this);
        const proto = getProto(target);
        const hadKey = proto.has.call(target, value);
        const result = proto.add.call(target, value);
        if (!hadKey) {
            trigger(target, "add" /* ADD */, value, value);
        }
        return result;
    }
    function set$1(key, value) {
        value = toRaw(value);
        const target = toRaw(this);
        const { has, get, set } = getProto(target);
        let hadKey = has.call(target, key);
        if (!hadKey) {
            key = toRaw(key);
            hadKey = has.call(target, key);
        }
        else {
            checkIdentityKeys(target, has, key);
        }
        const oldValue = get.call(target, key);
        const result = set.call(target, key, value);
        if (!hadKey) {
            trigger(target, "add" /* ADD */, key, value);
        }
        else if (hasChanged(value, oldValue)) {
            trigger(target, "set" /* SET */, key, value, oldValue);
        }
        return result;
    }
    function deleteEntry(key) {
        const target = toRaw(this);
        const { has, get, delete: del } = getProto(target);
        let hadKey = has.call(target, key);
        if (!hadKey) {
            key = toRaw(key);
            hadKey = has.call(target, key);
        }
        else {
            checkIdentityKeys(target, has, key);
        }
        const oldValue = get ? get.call(target, key) : undefined;
        // forward the operation before queueing reactions
        const result = del.call(target, key);
        if (hadKey) {
            trigger(target, "delete" /* DELETE */, key, undefined, oldValue);
        }
        return result;
    }
    function clear() {
        const target = toRaw(this);
        const hadItems = target.size !== 0;
        const oldTarget = target instanceof Map
                ? new Map(target)
                : new Set(target);
        // forward the operation before queueing reactions
        const result = getProto(target).clear.call(target);
        if (hadItems) {
            trigger(target, "clear" /* CLEAR */, undefined, undefined, oldTarget);
        }
        return result;
    }
    function createForEach(isReadonly, shallow) {
        return function forEach(callback, thisArg) {
            const observed = this;
            const target = toRaw(observed);
            const wrap = isReadonly ? toReadonly : shallow ? toShallow : toReactive;
            !isReadonly && track(target, "iterate" /* ITERATE */, ITERATE_KEY);
            // important: create sure the callback is
            // 1. invoked with the reactive map as `this` and 3rd arg
            // 2. the value received should be a corresponding reactive/readonly.
            function wrappedCallback(value, key) {
                return callback.call(thisArg, wrap(value), wrap(key), observed);
            }
            return getProto(target).forEach.call(target, wrappedCallback);
        };
    }
    function createIterableMethod(method, isReadonly, shallow) {
        return function (...args) {
            const target = toRaw(this);
            const isMap = target instanceof Map;
            const isPair = method === 'entries' || (method === Symbol.iterator && isMap);
            const isKeyOnly = method === 'keys' && isMap;
            const innerIterator = getProto(target)[method].apply(target, args);
            const wrap = isReadonly ? toReadonly : shallow ? toShallow : toReactive;
            !isReadonly &&
                track(target, "iterate" /* ITERATE */, isKeyOnly ? MAP_KEY_ITERATE_KEY : ITERATE_KEY);
            // return a wrapped iterator which returns observed versions of the
            // values emitted from the real iterator
            return {
                // iterator protocol
                next() {
                    const { value, done } = innerIterator.next();
                    return done
                        ? { value, done }
                        : {
                            value: isPair ? [wrap(value[0]), wrap(value[1])] : wrap(value),
                            done
                        };
                },
                // iterable protocol
                [Symbol.iterator]() {
                    return this;
                }
            };
        };
    }
    function createReadonlyMethod(type) {
        return function (...args) {
            {
                const key = args[0] ? `on key "${args[0]}" ` : ``;
                console.warn(`${capitalize(type)} operation ${key}failed: target is readonly.`, toRaw(this));
            }
            return type === "delete" /* DELETE */ ? false : this;
        };
    }
    const mutableInstrumentations = {
        get(key) {
            return get$1(this, key, toReactive);
        },
        get size() {
            return size(this);
        },
        has: has$1,
        add,
        set: set$1,
        delete: deleteEntry,
        clear,
        forEach: createForEach(false, false)
    };
    const shallowInstrumentations = {
        get(key) {
            return get$1(this, key, toShallow);
        },
        get size() {
            return size(this);
        },
        has: has$1,
        add,
        set: set$1,
        delete: deleteEntry,
        clear,
        forEach: createForEach(false, true)
    };
    const readonlyInstrumentations = {
        get(key) {
            return get$1(this, key, toReadonly);
        },
        get size() {
            return size(this);
        },
        has: has$1,
        add: createReadonlyMethod("add" /* ADD */),
        set: createReadonlyMethod("set" /* SET */),
        delete: createReadonlyMethod("delete" /* DELETE */),
        clear: createReadonlyMethod("clear" /* CLEAR */),
        forEach: createForEach(true, false)
    };
    const iteratorMethods = ['keys', 'values', 'entries', Symbol.iterator];
    iteratorMethods.forEach(method => {
        mutableInstrumentations[method] = createIterableMethod(method, false, false);
        readonlyInstrumentations[method] = createIterableMethod(method, true, false);
        shallowInstrumentations[method] = createIterableMethod(method, false, true);
    });
    function createInstrumentationGetter(isReadonly, shallow) {
        const instrumentations = shallow
            ? shallowInstrumentations
            : isReadonly
                ? readonlyInstrumentations
                : mutableInstrumentations;
        return (target, key, receiver) => {
            if (key === "__v_isReactive" /* IS_REACTIVE */) {
                return !isReadonly;
            }
            else if (key === "__v_isReadonly" /* IS_READONLY */) {
                return isReadonly;
            }
            else if (key === "__v_raw" /* RAW */) {
                return target;
            }
            return Reflect.get(hasOwn(instrumentations, key) && key in target
                ? instrumentations
                : target, key, receiver);
        };
    }
    const mutableCollectionHandlers = {
        get: createInstrumentationGetter(false, false)
    };
    const readonlyCollectionHandlers = {
        get: createInstrumentationGetter(true, false)
    };
    function checkIdentityKeys(target, has, key) {
        const rawKey = toRaw(key);
        if (rawKey !== key && has.call(target, rawKey)) {
            const type = toRawType(target);
            console.warn(`Reactive ${type} contains both the raw and reactive ` +
                `versions of the same object${type === `Map` ? `as keys` : ``}, ` +
                `which can lead to inconsistencies. ` +
                `Avoid differentiating between the raw and reactive versions ` +
                `of an object and only use the reactive version if possible.`);
        }
    }

    const collectionTypes = new Set([Set, Map, WeakMap, WeakSet]);
    const isObservableType = /*#__PURE__*/ makeMap('Object,Array,Map,Set,WeakMap,WeakSet');
    const canObserve = (value) => {
        return (!value["__v_skip" /* SKIP */] &&
            isObservableType(toRawType(value)) &&
            !Object.isFrozen(value));
    };
    function reactive(target) {
        // if trying to observe a readonly proxy, return the readonly version.
        if (target && target["__v_isReadonly" /* IS_READONLY */]) {
            return target;
        }
        return createReactiveObject(target, false, mutableHandlers, mutableCollectionHandlers);
    }
    function readonly(target) {
        return createReactiveObject(target, true, readonlyHandlers, readonlyCollectionHandlers);
    }
    // Return a reactive-copy of the original object, where only the root level
    // properties are readonly, and does NOT unwrap refs nor recursively convert
    // returned properties.
    // This is used for creating the props proxy object for stateful components.
    function shallowReadonly(target) {
        return createReactiveObject(target, true, shallowReadonlyHandlers, readonlyCollectionHandlers);
    }
    function createReactiveObject(target, isReadonly, baseHandlers, collectionHandlers) {
        if (!isObject(target)) {
            {
                console.warn(`value cannot be made reactive: ${String(target)}`);
            }
            return target;
        }
        // target is already a Proxy, return it.
        // exception: calling readonly() on a reactive object
        if (target["__v_raw" /* RAW */] &&
            !(isReadonly && target["__v_isReactive" /* IS_REACTIVE */])) {
            return target;
        }
        // target already has corresponding Proxy
        if (hasOwn(target, isReadonly ? "__v_readonly" /* READONLY */ : "__v_reactive" /* REACTIVE */)) {
            return isReadonly
                ? target["__v_readonly" /* READONLY */]
                : target["__v_reactive" /* REACTIVE */];
        }
        // only a whitelist of value types can be observed.
        if (!canObserve(target)) {
            return target;
        }
        const observed = new Proxy(target, collectionTypes.has(target.constructor) ? collectionHandlers : baseHandlers);
        def(target, isReadonly ? "__v_readonly" /* READONLY */ : "__v_reactive" /* REACTIVE */, observed);
        return observed;
    }
    function isReactive(value) {
        if (isReadonly(value)) {
            return isReactive(value["__v_raw" /* RAW */]);
        }
        return !!(value && value["__v_isReactive" /* IS_REACTIVE */]);
    }
    function isReadonly(value) {
        return !!(value && value["__v_isReadonly" /* IS_READONLY */]);
    }
    function isProxy(value) {
        return isReactive(value) || isReadonly(value);
    }
    function toRaw(observed) {
        return ((observed && toRaw(observed["__v_raw" /* RAW */])) || observed);
    }
    function isRef(r) {
        return r ? r.__v_isRef === true : false;
    }

    const stack = [];
    function pushWarningContext(vnode) {
        stack.push(vnode);
    }
    function popWarningContext() {
        stack.pop();
    }
    function warn(msg, ...args) {
        // avoid props formatting or warn handler tracking deps that might be mutated
        // during patch, leading to infinite recursion.
        pauseTracking();
        const instance = stack.length ? stack[stack.length - 1].component : null;
        const appWarnHandler = instance && instance.appContext.config.warnHandler;
        const trace = getComponentTrace();
        if (appWarnHandler) {
            callWithErrorHandling(appWarnHandler, instance, 11 /* APP_WARN_HANDLER */, [
                msg + args.join(''),
                instance && instance.proxy,
                trace
                    .map(({ vnode }) => `at <${formatComponentName(instance, vnode.type)}>`)
                    .join('\n'),
                trace
            ]);
        }
        else {
            const warnArgs = [`[Vue warn]: ${msg}`, ...args];
            if (trace.length &&
                // avoid spamming console during tests
                !false) {
                warnArgs.push(`\n`, ...formatTrace(trace));
            }
            console.warn(...warnArgs);
        }
        resetTracking();
    }
    function getComponentTrace() {
        let currentVNode = stack[stack.length - 1];
        if (!currentVNode) {
            return [];
        }
        // we can't just use the stack because it will be incomplete during updates
        // that did not start from the root. Re-construct the parent chain using
        // instance parent pointers.
        const normalizedStack = [];
        while (currentVNode) {
            const last = normalizedStack[0];
            if (last && last.vnode === currentVNode) {
                last.recurseCount++;
            }
            else {
                normalizedStack.push({
                    vnode: currentVNode,
                    recurseCount: 0
                });
            }
            const parentInstance = currentVNode.component && currentVNode.component.parent;
            currentVNode = parentInstance && parentInstance.vnode;
        }
        return normalizedStack;
    }
    function formatTrace(trace) {
        const logs = [];
        trace.forEach((entry, i) => {
            logs.push(...(i === 0 ? [] : [`\n`]), ...formatTraceEntry(entry));
        });
        return logs;
    }
    function formatTraceEntry({ vnode, recurseCount }) {
        const postfix = recurseCount > 0 ? `... (${recurseCount} recursive calls)` : ``;
        const isRoot = vnode.component ? vnode.component.parent == null : false;
        const open = ` at <${formatComponentName(vnode.component, vnode.type, isRoot)}`;
        const close = `>` + postfix;
        return vnode.props
            ? [open, ...formatProps(vnode.props), close]
            : [open + close];
    }
    function formatProps(props) {
        const res = [];
        const keys = Object.keys(props);
        keys.slice(0, 3).forEach(key => {
            res.push(...formatProp(key, props[key]));
        });
        if (keys.length > 3) {
            res.push(` ...`);
        }
        return res;
    }
    function formatProp(key, value, raw) {
        if (isString(value)) {
            value = JSON.stringify(value);
            return raw ? value : [`${key}=${value}`];
        }
        else if (typeof value === 'number' ||
            typeof value === 'boolean' ||
            value == null) {
            return raw ? value : [`${key}=${value}`];
        }
        else if (isRef(value)) {
            value = formatProp(key, toRaw(value.value), true);
            return raw ? value : [`${key}=Ref<`, value, `>`];
        }
        else if (isFunction(value)) {
            return [`${key}=fn${value.name ? `<${value.name}>` : ``}`];
        }
        else {
            value = toRaw(value);
            return raw ? value : [`${key}=`, value];
        }
    }

    const ErrorTypeStrings = {
        ["bc" /* BEFORE_CREATE */]: 'beforeCreate hook',
        ["c" /* CREATED */]: 'created hook',
        ["bm" /* BEFORE_MOUNT */]: 'beforeMount hook',
        ["m" /* MOUNTED */]: 'mounted hook',
        ["bu" /* BEFORE_UPDATE */]: 'beforeUpdate hook',
        ["u" /* UPDATED */]: 'updated',
        ["bum" /* BEFORE_UNMOUNT */]: 'beforeUnmount hook',
        ["um" /* UNMOUNTED */]: 'unmounted hook',
        ["a" /* ACTIVATED */]: 'activated hook',
        ["da" /* DEACTIVATED */]: 'deactivated hook',
        ["ec" /* ERROR_CAPTURED */]: 'errorCaptured hook',
        ["rtc" /* RENDER_TRACKED */]: 'renderTracked hook',
        ["rtg" /* RENDER_TRIGGERED */]: 'renderTriggered hook',
        [0 /* SETUP_FUNCTION */]: 'setup function',
        [1 /* RENDER_FUNCTION */]: 'render function',
        [2 /* WATCH_GETTER */]: 'watcher getter',
        [3 /* WATCH_CALLBACK */]: 'watcher callback',
        [4 /* WATCH_CLEANUP */]: 'watcher cleanup function',
        [5 /* NATIVE_EVENT_HANDLER */]: 'native event handler',
        [6 /* COMPONENT_EVENT_HANDLER */]: 'component event handler',
        [7 /* VNODE_HOOK */]: 'vnode hook',
        [8 /* DIRECTIVE_HOOK */]: 'directive hook',
        [9 /* TRANSITION_HOOK */]: 'transition hook',
        [10 /* APP_ERROR_HANDLER */]: 'app errorHandler',
        [11 /* APP_WARN_HANDLER */]: 'app warnHandler',
        [12 /* FUNCTION_REF */]: 'ref function',
        [13 /* ASYNC_COMPONENT_LOADER */]: 'async component loader',
        [14 /* SCHEDULER */]: 'scheduler flush. This is likely a Vue internals bug. ' +
            'Please open an issue at https://new-issue.vuejs.org/?repo=vuejs/vue-next'
    };
    function callWithErrorHandling(fn, instance, type, args) {
        let res;
        try {
            res = args ? fn(...args) : fn();
        }
        catch (err) {
            handleError(err, instance, type);
        }
        return res;
    }
    function callWithAsyncErrorHandling(fn, instance, type, args) {
        if (isFunction(fn)) {
            const res = callWithErrorHandling(fn, instance, type, args);
            if (res && isPromise(res)) {
                res.catch(err => {
                    handleError(err, instance, type);
                });
            }
            return res;
        }
        const values = [];
        for (let i = 0; i < fn.length; i++) {
            values.push(callWithAsyncErrorHandling(fn[i], instance, type, args));
        }
        return values;
    }
    function handleError(err, instance, type) {
        const contextVNode = instance ? instance.vnode : null;
        if (instance) {
            let cur = instance.parent;
            // the exposed instance is the render proxy to keep it consistent with 2.x
            const exposedInstance = instance.proxy;
            // in production the hook receives only the error code
            const errorInfo = ErrorTypeStrings[type];
            while (cur) {
                const errorCapturedHooks = cur.ec;
                if (errorCapturedHooks) {
                    for (let i = 0; i < errorCapturedHooks.length; i++) {
                        if (errorCapturedHooks[i](err, exposedInstance, errorInfo)) {
                            return;
                        }
                    }
                }
                cur = cur.parent;
            }
            // app-level handling
            const appErrorHandler = instance.appContext.config.errorHandler;
            if (appErrorHandler) {
                callWithErrorHandling(appErrorHandler, null, 10 /* APP_ERROR_HANDLER */, [err, exposedInstance, errorInfo]);
                return;
            }
        }
        logError(err, type, contextVNode);
    }
    function logError(err, type, contextVNode) {
        // default behavior is crash in prod & test, recover in dev.
        {
            const info = ErrorTypeStrings[type];
            if (contextVNode) {
                pushWarningContext(contextVNode);
            }
            warn(`Unhandled error${info ? ` during execution of ${info}` : ``}`);
            console.error(err);
            if (contextVNode) {
                popWarningContext();
            }
        }
    }

    const queue = [];
    const postFlushCbs = [];
    const p = Promise.resolve();
    let isFlushing = false;
    let isFlushPending = false;
    const RECURSION_LIMIT = 100;
    function nextTick(fn) {
        return fn ? p.then(fn) : p;
    }
    function queueJob(job) {
        if (!queue.includes(job)) {
            queue.push(job);
            queueFlush();
        }
    }
    function queuePostFlushCb(cb) {
        if (!isArray(cb)) {
            postFlushCbs.push(cb);
        }
        else {
            postFlushCbs.push(...cb);
        }
        queueFlush();
    }
    function queueFlush() {
        if (!isFlushing && !isFlushPending) {
            isFlushPending = true;
            nextTick(flushJobs);
        }
    }
    function flushPostFlushCbs(seen) {
        if (postFlushCbs.length) {
            const cbs = [...new Set(postFlushCbs)];
            postFlushCbs.length = 0;
            {
                seen = seen || new Map();
            }
            for (let i = 0; i < cbs.length; i++) {
                {
                    checkRecursiveUpdates(seen, cbs[i]);
                }
                cbs[i]();
            }
        }
    }
    const getId = (job) => (job.id == null ? Infinity : job.id);
    function flushJobs(seen) {
        isFlushPending = false;
        isFlushing = true;
        let job;
        {
            seen = seen || new Map();
        }
        // Sort queue before flush.
        // This ensures that:
        // 1. Components are updated from parent to child. (because parent is always
        //    created before the child so its render effect will have smaller
        //    priority number)
        // 2. If a component is unmounted during a parent component's update,
        //    its update can be skipped.
        // Jobs can never be null before flush starts, since they are only invalidated
        // during execution of another flushed job.
        queue.sort((a, b) => getId(a) - getId(b));
        while ((job = queue.shift()) !== undefined) {
            if (job === null) {
                continue;
            }
            {
                checkRecursiveUpdates(seen, job);
            }
            callWithErrorHandling(job, null, 14 /* SCHEDULER */);
        }
        flushPostFlushCbs(seen);
        isFlushing = false;
        // some postFlushCb queued jobs!
        // keep flushing until it drains.
        if (queue.length || postFlushCbs.length) {
            flushJobs(seen);
        }
    }
    function checkRecursiveUpdates(seen, fn) {
        if (!seen.has(fn)) {
            seen.set(fn, 1);
        }
        else {
            const count = seen.get(fn);
            if (count > RECURSION_LIMIT) {
                throw new Error('Maximum recursive updates exceeded. ' +
                    "You may have code that is mutating state in your component's " +
                    'render function or updated hook or watcher source function.');
            }
            else {
                seen.set(fn, count + 1);
            }
        }
    }
    const hmrDirtyComponents = new Set();
    // Expose the HMR runtime on the global object
    // This makes it entirely tree-shakable without polluting the exports and makes
    // it easier to be used in toolings like vue-loader
    // Note: for a component to be eligible for HMR it also needs the __hmrId option
    // to be set so that its instances can be registered / removed.
    {
        const globalObject = typeof global !== 'undefined'
            ? global
            : typeof self !== 'undefined'
                ? self
                : typeof window !== 'undefined'
                    ? window
                    : {};
        globalObject.__VUE_HMR_RUNTIME__ = {
            createRecord: tryWrap(createRecord),
            rerender: tryWrap(rerender),
            reload: tryWrap(reload)
        };
    }
    const map = new Map();
    function createRecord(id) {
        if (map.has(id)) {
            return false;
        }
        map.set(id, new Set());
        return true;
    }
    function rerender(id, newRender) {
        const record = map.get(id);
        if (!record)
            return;
        // Array.from creates a snapshot which avoids the set being mutated during
        // updates
        Array.from(record).forEach(instance => {
            if (newRender) {
                instance.render = newRender;
            }
            instance.renderCache = [];
            instance.update();
        });
    }
    function reload(id, newComp) {
        const record = map.get(id);
        if (!record)
            return;
        // Array.from creates a snapshot which avoids the set being mutated during
        // updates
        Array.from(record).forEach(instance => {
            const comp = instance.type;
            if (!hmrDirtyComponents.has(comp)) {
                // 1. Update existing comp definition to match new one
                extend(comp, newComp);
                for (const key in comp) {
                    if (!(key in newComp)) {
                        delete comp[key];
                    }
                }
                // 2. Mark component dirty. This forces the renderer to replace the component
                // on patch.
                hmrDirtyComponents.add(comp);
                // 3. Make sure to unmark the component after the reload.
                queuePostFlushCb(() => {
                    hmrDirtyComponents.delete(comp);
                });
            }
            if (instance.parent) {
                // 4. Force the parent instance to re-render. This will cause all updated
                // components to be unmounted and re-mounted. Queue the update so that we
                // don't end up forcing the same parent to re-render multiple times.
                queueJob(instance.parent.update);
            }
            else if (instance.appContext.reload) {
                // root instance mounted via createApp() has a reload method
                instance.appContext.reload();
            }
            else if (typeof window !== 'undefined') {
                // root instance inside tree created via raw render(). Force reload.
                window.location.reload();
            }
            else {
                console.warn('[HMR] Root or manually mounted instance modified. Full reload required.');
            }
        });
    }
    function tryWrap(fn) {
        return (id, arg) => {
            try {
                return fn(id, arg);
            }
            catch (e) {
                console.error(e);
                console.warn(`[HMR] Something went wrong during Vue component hot-reload. ` +
                    `Full reload required.`);
            }
        };
    }

    // mark the current rendering instance for asset resolution (e.g.
    // resolveComponent, resolveDirective) during render
    let currentRenderingInstance = null;

    const isSuspense = (type) => type.__isSuspense;
    function queueEffectWithSuspense(fn, suspense) {
        if (suspense && !suspense.isResolved) {
            if (isArray(fn)) {
                suspense.effects.push(...fn);
            }
            else {
                suspense.effects.push(fn);
            }
        }
        else {
            queuePostFlushCb(fn);
        }
    }

    // SFC scoped style ID management.
    let currentScopeId = null;

    const isTeleport = (type) => type.__isTeleport;
    const NULL_DYNAMIC_COMPONENT = Symbol();

    const Fragment = Symbol('Fragment');
    const Text = Symbol('Text');
    const Comment = Symbol('Comment');
    let currentBlock = null;
    // Whether we should be tracking dynamic child nodes inside a block.
    // Only tracks when this value is > 0
    // We are not using a simple boolean because this value may need to be
    // incremented/decremented by nested usage of v-once (see below)
    let shouldTrack$1 = 1;
    function isVNode(value) {
        return value ? value.__v_isVNode === true : false;
    }
    function isSameVNodeType(n1, n2) {
        if (n2.shapeFlag & 6 /* COMPONENT */ &&
            hmrDirtyComponents.has(n2.type)) {
            // HMR only: if the component has been hot-updated, force a reload.
            return false;
        }
        return n1.type === n2.type && n1.key === n2.key;
    }
    let vnodeArgsTransformer;
    const createVNodeWithArgsTransform = (...args) => {
        return _createVNode(...(vnodeArgsTransformer
            ? vnodeArgsTransformer(args, currentRenderingInstance)
            : args));
    };
    const InternalObjectKey = `__vInternal`;
    const normalizeKey = ({ key }) => key != null ? key : null;
    const normalizeRef = ({ ref: ref$$1 }) => {
        return (ref$$1 != null
            ? isArray(ref$$1)
                ? ref$$1
                : [currentRenderingInstance, ref$$1]
            : null);
    };
    const createVNode = (createVNodeWithArgsTransform);
    function _createVNode(type, props = null, children = null, patchFlag = 0, dynamicProps = null, isBlockNode = false) {
        if (!type || type === NULL_DYNAMIC_COMPONENT) {
            if (!type) {
                warn(`Invalid vnode type when creating vnode: ${type}.`);
            }
            type = Comment;
        }
        if (isVNode(type)) {
            return cloneVNode(type, props, children);
        }
        // class component normalization.
        if (isFunction(type) && '__vccOpts' in type) {
            type = type.__vccOpts;
        }
        // class & style normalization.
        if (props) {
            // for reactive or proxy objects, we need to clone it to enable mutation.
            if (isProxy(props) || InternalObjectKey in props) {
                props = extend({}, props);
            }
            let { class: klass, style } = props;
            if (klass && !isString(klass)) {
                props.class = normalizeClass(klass);
            }
            if (isObject(style)) {
                // reactive state objects need to be cloned since they are likely to be
                // mutated
                if (isProxy(style) && !isArray(style)) {
                    style = extend({}, style);
                }
                props.style = normalizeStyle(style);
            }
        }
        // encode the vnode type information into a bitmap
        const shapeFlag = isString(type)
            ? 1 /* ELEMENT */
            :  isSuspense(type)
                ? 128 /* SUSPENSE */
                : isTeleport(type)
                    ? 64 /* TELEPORT */
                    : isObject(type)
                        ? 4 /* STATEFUL_COMPONENT */
                        : isFunction(type)
                            ? 2 /* FUNCTIONAL_COMPONENT */
                            : 0;
        if (shapeFlag & 4 /* STATEFUL_COMPONENT */ && isProxy(type)) {
            type = toRaw(type);
            warn(`Vue received a Component which was made a reactive object. This can ` +
                `lead to unnecessary performance overhead, and should be avoided by ` +
                `marking the component with \`markRaw\` or using \`shallowRef\` ` +
                `instead of \`ref\`.`, `\nComponent that was made reactive: `, type);
        }
        const vnode = {
            __v_isVNode: true,
            __v_skip: true,
            type,
            props,
            key: props && normalizeKey(props),
            ref: props && normalizeRef(props),
            scopeId: currentScopeId,
            children: null,
            component: null,
            suspense: null,
            dirs: null,
            transition: null,
            el: null,
            anchor: null,
            target: null,
            targetAnchor: null,
            staticCount: 0,
            shapeFlag,
            patchFlag,
            dynamicProps,
            dynamicChildren: null,
            appContext: null
        };
        // validate key
        if (vnode.key !== vnode.key) {
            warn(`VNode created with invalid key (NaN). VNode type:`, vnode.type);
        }
        normalizeChildren(vnode, children);
        // presence of a patch flag indicates this node needs patching on updates.
        // component nodes also should always be patched, because even if the
        // component doesn't need to update, it needs to persist the instance on to
        // the next vnode so that it can be properly unmounted later.
        if (shouldTrack$1 > 0 &&
            !isBlockNode &&
            currentBlock &&
            // the EVENTS flag is only for hydration and if it is the only flag, the
            // vnode should not be considered dynamic due to handler caching.
            patchFlag !== 32 /* HYDRATE_EVENTS */ &&
            (patchFlag > 0 ||
                shapeFlag & 128 /* SUSPENSE */ ||
                shapeFlag & 64 /* TELEPORT */ ||
                shapeFlag & 4 /* STATEFUL_COMPONENT */ ||
                shapeFlag & 2 /* FUNCTIONAL_COMPONENT */)) {
            currentBlock.push(vnode);
        }
        return vnode;
    }
    function cloneVNode(vnode, extraProps, children) {
        const props = extraProps
            ? vnode.props
                ? mergeProps(vnode.props, extraProps)
                : extend({}, extraProps)
            : vnode.props;
        // This is intentionally NOT using spread or extend to avoid the runtime
        // key enumeration cost.
        const cloned = {
            __v_isVNode: true,
            __v_skip: true,
            type: vnode.type,
            props,
            key: props && normalizeKey(props),
            ref: extraProps && extraProps.ref ? normalizeRef(extraProps) : vnode.ref,
            scopeId: vnode.scopeId,
            children: vnode.children,
            target: vnode.target,
            targetAnchor: vnode.targetAnchor,
            staticCount: vnode.staticCount,
            shapeFlag: vnode.shapeFlag,
            // if the vnode is cloned with extra props, we can no longer assume its
            // existing patch flag to be reliable and need to bail out of optimized mode.
            // however we don't want block nodes to de-opt their children, so if the
            // vnode is a block node, we only add the FULL_PROPS flag to it.
            patchFlag: extraProps
                ? vnode.dynamicChildren
                    ? vnode.patchFlag | 16 /* FULL_PROPS */
                    : -2 /* BAIL */
                : vnode.patchFlag,
            dynamicProps: vnode.dynamicProps,
            dynamicChildren: vnode.dynamicChildren,
            appContext: vnode.appContext,
            dirs: vnode.dirs,
            transition: vnode.transition,
            // These should technically only be non-null on mounted VNodes. However,
            // they *should* be copied for kept-alive vnodes. So we just always copy
            // them since them being non-null during a mount doesn't affect the logic as
            // they will simply be overwritten.
            component: vnode.component,
            suspense: vnode.suspense,
            el: vnode.el,
            anchor: vnode.anchor
        };
        if (children) {
            normalizeChildren(cloned, children);
        }
        return cloned;
    }
    /**
     * @private
     */
    function createTextVNode(text = ' ', flag = 0) {
        return createVNode(Text, null, text, flag);
    }
    function normalizeChildren(vnode, children) {
        let type = 0;
        const { shapeFlag } = vnode;
        if (children == null) {
            children = null;
        }
        else if (isArray(children)) {
            type = 16 /* ARRAY_CHILDREN */;
        }
        else if (typeof children === 'object') {
            // Normalize slot to plain children
            if ((shapeFlag & 1 /* ELEMENT */ || shapeFlag & 64 /* TELEPORT */) &&
                children.default) {
                normalizeChildren(vnode, children.default());
                return;
            }
            else {
                type = 32 /* SLOTS_CHILDREN */;
                if (!children._ && !(InternalObjectKey in children)) {
                    children._ctx = currentRenderingInstance;
                }
            }
        }
        else if (isFunction(children)) {
            children = { default: children, _ctx: currentRenderingInstance };
            type = 32 /* SLOTS_CHILDREN */;
        }
        else {
            children = String(children);
            // force teleport children to array so it can be moved around
            if (shapeFlag & 64 /* TELEPORT */) {
                type = 16 /* ARRAY_CHILDREN */;
                children = [createTextVNode(children)];
            }
            else {
                type = 8 /* TEXT_CHILDREN */;
            }
        }
        vnode.children = children;
        vnode.shapeFlag |= type;
    }
    const handlersRE = /^on|^vnode/;
    function mergeProps(...args) {
        const ret = extend({}, args[0]);
        for (let i = 1; i < args.length; i++) {
            const toMerge = args[i];
            for (const key in toMerge) {
                if (key === 'class') {
                    if (ret.class !== toMerge.class) {
                        ret.class = normalizeClass([ret.class, toMerge.class]);
                    }
                }
                else if (key === 'style') {
                    ret.style = normalizeStyle([ret.style, toMerge.style]);
                }
                else if (handlersRE.test(key)) {
                    // on*, vnode*
                    const existing = ret[key];
                    const incoming = toMerge[key];
                    if (existing !== incoming) {
                        ret[key] = existing
                            ? [].concat(existing, toMerge[key])
                            : incoming;
                    }
                }
                else {
                    ret[key] = toMerge[key];
                }
            }
        }
        return ret;
    }
    function normalizePropsOptions(comp) {
        if (comp.__props) {
            return comp.__props;
        }
        const raw = comp.props;
        const normalized = {};
        const needCastKeys = [];
        // apply mixin/extends props
        let hasExtends = false;
        if ( !isFunction(comp)) {
            const extendProps = (raw) => {
                const [props, keys] = normalizePropsOptions(raw);
                extend(normalized, props);
                if (keys)
                    needCastKeys.push(...keys);
            };
            if (comp.extends) {
                hasExtends = true;
                extendProps(comp.extends);
            }
            if (comp.mixins) {
                hasExtends = true;
                comp.mixins.forEach(extendProps);
            }
        }
        if (!raw && !hasExtends) {
            return (comp.__props = EMPTY_ARR);
        }
        if (isArray(raw)) {
            for (let i = 0; i < raw.length; i++) {
                if (!isString(raw[i])) {
                    warn(`props must be strings when using array syntax.`, raw[i]);
                }
                const normalizedKey = camelize(raw[i]);
                if (validatePropName(normalizedKey)) {
                    normalized[normalizedKey] = EMPTY_OBJ;
                }
            }
        }
        else if (raw) {
            if (!isObject(raw)) {
                warn(`invalid props options`, raw);
            }
            for (const key in raw) {
                const normalizedKey = camelize(key);
                if (validatePropName(normalizedKey)) {
                    const opt = raw[key];
                    const prop = (normalized[normalizedKey] =
                        isArray(opt) || isFunction(opt) ? { type: opt } : opt);
                    if (prop) {
                        const booleanIndex = getTypeIndex(Boolean, prop.type);
                        const stringIndex = getTypeIndex(String, prop.type);
                        prop[0 /* shouldCast */] = booleanIndex > -1;
                        prop[1 /* shouldCastTrue */] =
                            stringIndex < 0 || booleanIndex < stringIndex;
                        // if the prop needs boolean casting or default value
                        if (booleanIndex > -1 || hasOwn(prop, 'default')) {
                            needCastKeys.push(normalizedKey);
                        }
                    }
                }
            }
        }
        const normalizedEntry = [normalized, needCastKeys];
        comp.__props = normalizedEntry;
        return normalizedEntry;
    }
    // use function string name to check type constructors
    // so that it works across vms / iframes.
    function getType(ctor) {
        const match = ctor && ctor.toString().match(/^\s*function (\w+)/);
        return match ? match[1] : '';
    }
    function isSameType(a, b) {
        return getType(a) === getType(b);
    }
    function getTypeIndex(type, expectedTypes) {
        if (isArray(expectedTypes)) {
            for (let i = 0, len = expectedTypes.length; i < len; i++) {
                if (isSameType(expectedTypes[i], type)) {
                    return i;
                }
            }
        }
        else if (isFunction(expectedTypes)) {
            return isSameType(expectedTypes, type) ? 0 : -1;
        }
        return -1;
    }
    /**
     * dev only
     */
    function validatePropName(key) {
        if (key[0] !== '$') {
            return true;
        }
        else {
            warn(`Invalid prop name: "${key}" is a reserved property.`);
        }
        return false;
    }
    const isSimpleType = /*#__PURE__*/ makeMap('String,Number,Boolean,Function,Symbol');

    function injectHook(type, hook, target = currentInstance, prepend = false) {
        if (target) {
            const hooks = target[type] || (target[type] = []);
            // cache the error handling wrapper for injected hooks so the same hook
            // can be properly deduped by the scheduler. "__weh" stands for "with error
            // handling".
            const wrappedHook = hook.__weh ||
                (hook.__weh = (...args) => {
                    if (target.isUnmounted) {
                        return;
                    }
                    // disable tracking inside all lifecycle hooks
                    // since they can potentially be called inside effects.
                    pauseTracking();
                    // Set currentInstance during hook invocation.
                    // This assumes the hook does not synchronously trigger other hooks, which
                    // can only be false when the user does something really funky.
                    setCurrentInstance(target);
                    const res = callWithAsyncErrorHandling(hook, target, type, args);
                    setCurrentInstance(null);
                    resetTracking();
                    return res;
                });
            if (prepend) {
                hooks.unshift(wrappedHook);
            }
            else {
                hooks.push(wrappedHook);
            }
        }
        else {
            const apiName = `on${capitalize(ErrorTypeStrings[type].replace(/ hook$/, ''))}`;
            warn(`${apiName} is called when there is no active component instance to be ` +
                `associated with. ` +
                `Lifecycle injection APIs can only be used during execution of setup().` +
                ( ` If you are using async setup(), make sure to register lifecycle ` +
                        `hooks before the first await statement.`
                    ));
        }
    }
    const createHook = (lifecycle) => (hook, target = currentInstance) => 
    // post-create lifecycle registrations are noops during SSR
    !isInSSRComponentSetup && injectHook(lifecycle, hook, target);
    const onMounted = createHook("m" /* MOUNTED */);
    const onUpdated = createHook("u" /* UPDATED */);
    const onBeforeUnmount = createHook("bum" /* BEFORE_UNMOUNT */);

    function useTransitionState() {
        const state = {
            isMounted: false,
            isLeaving: false,
            isUnmounting: false,
            leavingVNodes: new Map()
        };
        onMounted(() => {
            state.isMounted = true;
        });
        onBeforeUnmount(() => {
            state.isUnmounting = true;
        });
        return state;
    }
    const BaseTransitionImpl = {
        name: `BaseTransition`,
        props: {
            mode: String,
            appear: Boolean,
            persisted: Boolean,
            // enter
            onBeforeEnter: Function,
            onEnter: Function,
            onAfterEnter: Function,
            onEnterCancelled: Function,
            // leave
            onBeforeLeave: Function,
            onLeave: Function,
            onAfterLeave: Function,
            onLeaveCancelled: Function,
            // appear
            onBeforeAppear: Function,
            onAppear: Function,
            onAfterAppear: Function,
            onAppearCancelled: Function
        },
        setup(props, { slots }) {
            const instance = getCurrentInstance();
            const state = useTransitionState();
            let prevTransitionKey;
            return () => {
                const children = slots.default && getTransitionRawChildren(slots.default(), true);
                if (!children || !children.length) {
                    return;
                }
                // warn multiple elements
                if (children.length > 1) {
                    warn('<transition> can only be used on a single element or component. Use ' +
                        '<transition-group> for lists.');
                }
                // there's no need to track reactivity for these props so use the raw
                // props for a bit better perf
                const rawProps = toRaw(props);
                const { mode } = rawProps;
                // check mode
                if (mode && !['in-out', 'out-in', 'default'].includes(mode)) {
                    warn(`invalid <transition> mode: ${mode}`);
                }
                // at this point children has a guaranteed length of 1.
                const child = children[0];
                if (state.isLeaving) {
                    return emptyPlaceholder(child);
                }
                // in the case of <transition><keep-alive/></transition>, we need to
                // compare the type of the kept-alive children.
                const innerChild = getKeepAliveChild(child);
                if (!innerChild) {
                    return emptyPlaceholder(child);
                }
                const enterHooks = (innerChild.transition = resolveTransitionHooks(innerChild, rawProps, state, instance));
                const oldChild = instance.subTree;
                const oldInnerChild = oldChild && getKeepAliveChild(oldChild);
                let transitionKeyChanged = false;
                const { getTransitionKey } = innerChild.type;
                if (getTransitionKey) {
                    const key = getTransitionKey();
                    if (prevTransitionKey === undefined) {
                        prevTransitionKey = key;
                    }
                    else if (key !== prevTransitionKey) {
                        prevTransitionKey = key;
                        transitionKeyChanged = true;
                    }
                }
                // handle mode
                if (oldInnerChild &&
                    oldInnerChild.type !== Comment &&
                    (!isSameVNodeType(innerChild, oldInnerChild) || transitionKeyChanged)) {
                    const leavingHooks = resolveTransitionHooks(oldInnerChild, rawProps, state, instance);
                    // update old tree's hooks in case of dynamic transition
                    setTransitionHooks(oldInnerChild, leavingHooks);
                    // switching between different views
                    if (mode === 'out-in') {
                        state.isLeaving = true;
                        // return placeholder node and queue update when leave finishes
                        leavingHooks.afterLeave = () => {
                            state.isLeaving = false;
                            instance.update();
                        };
                        return emptyPlaceholder(child);
                    }
                    else if (mode === 'in-out') {
                        leavingHooks.delayLeave = (el, earlyRemove, delayedLeave) => {
                            const leavingVNodesCache = getLeavingNodesForType(state, oldInnerChild);
                            leavingVNodesCache[String(oldInnerChild.key)] = oldInnerChild;
                            // early removal callback
                            el._leaveCb = () => {
                                earlyRemove();
                                el._leaveCb = undefined;
                                delete enterHooks.delayedLeave;
                            };
                            enterHooks.delayedLeave = delayedLeave;
                        };
                    }
                }
                return child;
            };
        }
    };
    // export the public type for h/tsx inference
    // also to avoid inline import() in generated d.ts files
    const BaseTransition = BaseTransitionImpl;
    function getLeavingNodesForType(state, vnode) {
        const { leavingVNodes } = state;
        let leavingVNodesCache = leavingVNodes.get(vnode.type);
        if (!leavingVNodesCache) {
            leavingVNodesCache = Object.create(null);
            leavingVNodes.set(vnode.type, leavingVNodesCache);
        }
        return leavingVNodesCache;
    }
    // The transition hooks are attached to the vnode as vnode.transition
    // and will be called at appropriate timing in the renderer.
    function resolveTransitionHooks(vnode, { appear, persisted = false, onBeforeEnter, onEnter, onAfterEnter, onEnterCancelled, onBeforeLeave, onLeave, onAfterLeave, onLeaveCancelled, onBeforeAppear, onAppear, onAfterAppear, onAppearCancelled }, state, instance) {
        const key = String(vnode.key);
        const leavingVNodesCache = getLeavingNodesForType(state, vnode);
        const callHook = (hook, args) => {
            hook &&
                callWithAsyncErrorHandling(hook, instance, 9 /* TRANSITION_HOOK */, args);
        };
        const hooks = {
            persisted,
            beforeEnter(el) {
                let hook = onBeforeEnter;
                if (!state.isMounted) {
                    if (appear) {
                        hook = onBeforeAppear || onBeforeEnter;
                    }
                    else {
                        return;
                    }
                }
                // for same element (v-show)
                if (el._leaveCb) {
                    el._leaveCb(true /* cancelled */);
                }
                // for toggled element with same key (v-if)
                const leavingVNode = leavingVNodesCache[key];
                if (leavingVNode &&
                    isSameVNodeType(vnode, leavingVNode) &&
                    leavingVNode.el._leaveCb) {
                    // force early removal (not cancelled)
                    leavingVNode.el._leaveCb();
                }
                callHook(hook, [el]);
            },
            enter(el) {
                let hook = onEnter;
                let afterHook = onAfterEnter;
                let cancelHook = onEnterCancelled;
                if (!state.isMounted) {
                    if (appear) {
                        hook = onAppear || onEnter;
                        afterHook = onAfterAppear || onAfterEnter;
                        cancelHook = onAppearCancelled || onEnterCancelled;
                    }
                    else {
                        return;
                    }
                }
                let called = false;
                const done = (el._enterCb = (cancelled) => {
                    if (called)
                        return;
                    called = true;
                    if (cancelled) {
                        callHook(cancelHook, [el]);
                    }
                    else {
                        callHook(afterHook, [el]);
                    }
                    if (hooks.delayedLeave) {
                        hooks.delayedLeave();
                    }
                    el._enterCb = undefined;
                });
                if (hook) {
                    hook(el, done);
                    if (hook.length <= 1) {
                        done();
                    }
                }
                else {
                    done();
                }
            },
            leave(el, remove$$1) {
                const key = String(vnode.key);
                if (el._enterCb) {
                    el._enterCb(true /* cancelled */);
                }
                if (state.isUnmounting) {
                    return remove$$1();
                }
                callHook(onBeforeLeave, [el]);
                let called = false;
                const done = (el._leaveCb = (cancelled) => {
                    if (called)
                        return;
                    called = true;
                    remove$$1();
                    if (cancelled) {
                        callHook(onLeaveCancelled, [el]);
                    }
                    else {
                        callHook(onAfterLeave, [el]);
                    }
                    el._leaveCb = undefined;
                    if (leavingVNodesCache[key] === vnode) {
                        delete leavingVNodesCache[key];
                    }
                });
                leavingVNodesCache[key] = vnode;
                if (onLeave) {
                    onLeave(el, done);
                    if (onLeave.length <= 1) {
                        done();
                    }
                }
                else {
                    done();
                }
            }
        };
        return hooks;
    }
    // the placeholder really only handles one special case: KeepAlive
    // in the case of a KeepAlive in a leave phase we need to return a KeepAlive
    // placeholder with empty content to avoid the KeepAlive instance from being
    // unmounted.
    function emptyPlaceholder(vnode) {
        if (isKeepAlive(vnode)) {
            vnode = cloneVNode(vnode);
            vnode.children = null;
            return vnode;
        }
    }
    function getKeepAliveChild(vnode) {
        return isKeepAlive(vnode)
            ? vnode.children
                ? vnode.children[0]
                : undefined
            : vnode;
    }
    function setTransitionHooks(vnode, hooks) {
        if (vnode.shapeFlag & 6 /* COMPONENT */ && vnode.component) {
            setTransitionHooks(vnode.component.subTree, hooks);
        }
        else {
            vnode.transition = hooks;
        }
    }
    function getTransitionRawChildren(children, keepComment = false) {
        let ret = [];
        let keyedFragmentCount = 0;
        for (let i = 0; i < children.length; i++) {
            const child = children[i];
            // handle fragment children case, e.g. v-for
            if (child.type === Fragment) {
                if (child.patchFlag & 128 /* KEYED_FRAGMENT */)
                    keyedFragmentCount++;
                ret = ret.concat(getTransitionRawChildren(child.children, keepComment));
            }
            // comment placeholders should be skipped, e.g. v-if
            else if (keepComment || child.type !== Comment) {
                ret.push(child);
            }
        }
        // #1126 if a transition children list contains multiple sub fragments, these
        // fragments will be merged into a flat children array. Since each v-for
        // fragment may contain different static bindings inside, we need to de-top
        // these children to force full diffs to ensure correct behavior.
        if (keyedFragmentCount > 1) {
            for (let i = 0; i < ret.length; i++) {
                ret[i].patchFlag = -2 /* BAIL */;
            }
        }
        return ret;
    }

    const isKeepAlive = (vnode) => vnode.type.__isKeepAlive;

    /**
    Runtime helper for applying directives to a vnode. Example usage:

    const comp = resolveComponent('comp')
    const foo = resolveDirective('foo')
    const bar = resolveDirective('bar')

    return withDirectives(h(comp), [
      [foo, this.x],
      [bar, this.y]
    ])
    */
    const isBuiltInDirective = /*#__PURE__*/ makeMap('bind,cloak,else-if,else,for,html,if,model,on,once,pre,show,slot,text');
    const queuePostRenderEffect =  queueEffectWithSuspense
        ;

    const invoke = (fn) => fn();
    // initial value for watchers to trigger on undefined initial values
    const INITIAL_WATCHER_VALUE = {};
    function doWatch(source, cb, { immediate, deep, flush, onTrack, onTrigger } = EMPTY_OBJ, instance = currentInstance) {
        if (!cb) {
            if (immediate !== undefined) {
                warn(`watch() "immediate" option is only respected when using the ` +
                    `watch(source, callback, options?) signature.`);
            }
            if (deep !== undefined) {
                warn(`watch() "deep" option is only respected when using the ` +
                    `watch(source, callback, options?) signature.`);
            }
        }
        const warnInvalidSource = (s) => {
            warn(`Invalid watch source: `, s, `A watch source can only be a getter/effect function, a ref, ` +
                `a reactive object, or an array of these types.`);
        };
        let getter;
        if (isArray(source)) {
            getter = () => source.map(s => {
                if (isRef(s)) {
                    return s.value;
                }
                else if (isReactive(s)) {
                    return traverse(s);
                }
                else if (isFunction(s)) {
                    return callWithErrorHandling(s, instance, 2 /* WATCH_GETTER */);
                }
                else {
                    warnInvalidSource(s);
                }
            });
        }
        else if (isRef(source)) {
            getter = () => source.value;
        }
        else if (isReactive(source)) {
            getter = () => source;
            deep = true;
        }
        else if (isFunction(source)) {
            if (cb) {
                // getter with cb
                getter = () => callWithErrorHandling(source, instance, 2 /* WATCH_GETTER */);
            }
            else {
                // no cb -> simple effect
                getter = () => {
                    if (instance && instance.isUnmounted) {
                        return;
                    }
                    if (cleanup) {
                        cleanup();
                    }
                    return callWithErrorHandling(source, instance, 3 /* WATCH_CALLBACK */, [onInvalidate]);
                };
            }
        }
        else {
            getter = NOOP;
            warnInvalidSource(source);
        }
        if (cb && deep) {
            const baseGetter = getter;
            getter = () => traverse(baseGetter());
        }
        let cleanup;
        const onInvalidate = (fn) => {
            cleanup = runner.options.onStop = () => {
                callWithErrorHandling(fn, instance, 4 /* WATCH_CLEANUP */);
            };
        };
        let oldValue = isArray(source) ? [] : INITIAL_WATCHER_VALUE;
        const applyCb = cb
            ? () => {
                if (instance && instance.isUnmounted) {
                    return;
                }
                const newValue = runner();
                if (deep || hasChanged(newValue, oldValue)) {
                    // cleanup before running cb again
                    if (cleanup) {
                        cleanup();
                    }
                    callWithAsyncErrorHandling(cb, instance, 3 /* WATCH_CALLBACK */, [
                        newValue,
                        // pass undefined as the old value when it's changed for the first time
                        oldValue === INITIAL_WATCHER_VALUE ? undefined : oldValue,
                        onInvalidate
                    ]);
                    oldValue = newValue;
                }
            }
            : void 0;
        let scheduler;
        if (flush === 'sync') {
            scheduler = invoke;
        }
        else if (flush === 'pre') {
            scheduler = job => {
                if (!instance || instance.isMounted) {
                    queueJob(job);
                }
                else {
                    // with 'pre' option, the first call must happen before
                    // the component is mounted so it is called synchronously.
                    job();
                }
            };
        }
        else {
            scheduler = job => queuePostRenderEffect(job, instance && instance.suspense);
        }
        const runner = effect(getter, {
            lazy: true,
            // so it runs before component update effects in pre flush mode
            computed: true,
            onTrack,
            onTrigger,
            scheduler: applyCb ? () => scheduler(applyCb) : scheduler
        });
        recordInstanceBoundEffect(runner);
        // initial run
        if (applyCb) {
            if (immediate) {
                applyCb();
            }
            else {
                oldValue = runner();
            }
        }
        else {
            runner();
        }
        return () => {
            stop(runner);
            if (instance) {
                remove(instance.effects, runner);
            }
        };
    }
    // this.$watch
    function instanceWatch(source, cb, options) {
        const publicThis = this.proxy;
        const getter = isString(source)
            ? () => publicThis[source]
            : source.bind(publicThis);
        return doWatch(getter, cb.bind(publicThis), options, this);
    }
    function traverse(value, seen = new Set()) {
        if (!isObject(value) || seen.has(value)) {
            return value;
        }
        seen.add(value);
        if (isArray(value)) {
            for (let i = 0; i < value.length; i++) {
                traverse(value[i], seen);
            }
        }
        else if (value instanceof Map) {
            value.forEach((v, key) => {
                // to register mutation dep for existing keys
                traverse(value.get(key), seen);
            });
        }
        else if (value instanceof Set) {
            value.forEach(v => {
                traverse(v, seen);
            });
        }
        else {
            for (const key in value) {
                traverse(value[key], seen);
            }
        }
        return value;
    }
    function resolveMergedOptions(instance) {
        const raw = instance.type;
        const { __merged, mixins, extends: extendsOptions } = raw;
        if (__merged)
            return __merged;
        const globalMixins = instance.appContext.mixins;
        if (!globalMixins.length && !mixins && !extendsOptions)
            return raw;
        const options = {};
        globalMixins.forEach(m => mergeOptions(options, m, instance));
        extendsOptions && mergeOptions(options, extendsOptions, instance);
        mixins && mixins.forEach(m => mergeOptions(options, m, instance));
        mergeOptions(options, raw, instance);
        return (raw.__merged = options);
    }
    function mergeOptions(to, from, instance) {
        const strats = instance.appContext.config.optionMergeStrategies;
        for (const key in from) {
            if (strats && hasOwn(strats, key)) {
                to[key] = strats[key](to[key], from[key], instance.proxy, key);
            }
            else if (!hasOwn(to, key)) {
                to[key] = from[key];
            }
        }
    }

    const publicPropertiesMap = extend(Object.create(null), {
        $: i => i,
        $el: i => i.vnode.el,
        $data: i => i.data,
        $props: i => (shallowReadonly(i.props)),
        $attrs: i => (shallowReadonly(i.attrs)),
        $slots: i => (shallowReadonly(i.slots)),
        $refs: i => (shallowReadonly(i.refs)),
        $parent: i => i.parent && i.parent.proxy,
        $root: i => i.root && i.root.proxy,
        $emit: i => i.emit,
        $options: i => ( resolveMergedOptions(i) ),
        $forceUpdate: i => () => queueJob(i.update),
        $nextTick: () => nextTick,
        $watch:  i => instanceWatch.bind(i) 
    });
    const PublicInstanceProxyHandlers = {
        get({ _: instance }, key) {
            const { ctx, setupState, data, props, accessCache, type, appContext } = instance;
            // let @vue/reactivity know it should never observe Vue public instances.
            if (key === "__v_skip" /* SKIP */) {
                return true;
            }
            // data / props / ctx
            // This getter gets called for every property access on the render context
            // during render and is a major hotspot. The most expensive part of this
            // is the multiple hasOwn() calls. It's much faster to do a simple property
            // access on a plain object, so we use an accessCache object (with null
            // prototype) to memoize what access type a key corresponds to.
            let normalizedProps;
            if (key[0] !== '$') {
                const n = accessCache[key];
                if (n !== undefined) {
                    switch (n) {
                        case 0 /* SETUP */:
                            return setupState[key];
                        case 1 /* DATA */:
                            return data[key];
                        case 3 /* CONTEXT */:
                            return ctx[key];
                        case 2 /* PROPS */:
                            return props[key];
                        // default: just fallthrough
                    }
                }
                else if (setupState !== EMPTY_OBJ && hasOwn(setupState, key)) {
                    accessCache[key] = 0 /* SETUP */;
                    return setupState[key];
                }
                else if (data !== EMPTY_OBJ && hasOwn(data, key)) {
                    accessCache[key] = 1 /* DATA */;
                    return data[key];
                }
                else if (
                // only cache other properties when instance has declared (thus stable)
                // props
                (normalizedProps = normalizePropsOptions(type)[0]) &&
                    hasOwn(normalizedProps, key)) {
                    accessCache[key] = 2 /* PROPS */;
                    return props[key];
                }
                else if (ctx !== EMPTY_OBJ && hasOwn(ctx, key)) {
                    accessCache[key] = 3 /* CONTEXT */;
                    return ctx[key];
                }
                else {
                    accessCache[key] = 4 /* OTHER */;
                }
            }
            const publicGetter = publicPropertiesMap[key];
            let cssModule, globalProperties;
            // public $xxx properties
            if (publicGetter) {
                if (key === '$attrs') {
                    track(instance, "get" /* GET */, key);
                }
                return publicGetter(instance);
            }
            else if (
            // css module (injected by vue-loader)
            (cssModule = type.__cssModules) &&
                (cssModule = cssModule[key])) {
                return cssModule;
            }
            else if (ctx !== EMPTY_OBJ && hasOwn(ctx, key)) {
                // user may set custom properties to `this` that start with `$`
                accessCache[key] = 3 /* CONTEXT */;
                return ctx[key];
            }
            else if (
            // global properties
            ((globalProperties = appContext.config.globalProperties),
                hasOwn(globalProperties, key))) {
                return globalProperties[key];
            }
            else if (currentRenderingInstance &&
                // #1091 avoid internal isRef/isVNode checks on component instance leading
                // to infinite warning loop
                key.indexOf('__v') !== 0) {
                if (data !== EMPTY_OBJ && key[0] === '$' && hasOwn(data, key)) {
                    warn(`Property ${JSON.stringify(key)} must be accessed via $data because it starts with a reserved ` +
                        `character and is not proxied on the render context.`);
                }
                else {
                    warn(`Property ${JSON.stringify(key)} was accessed during render ` +
                        `but is not defined on instance.`);
                }
            }
        },
        set({ _: instance }, key, value) {
            const { data, setupState, ctx } = instance;
            if (setupState !== EMPTY_OBJ && hasOwn(setupState, key)) {
                setupState[key] = value;
            }
            else if (data !== EMPTY_OBJ && hasOwn(data, key)) {
                data[key] = value;
            }
            else if (key in instance.props) {
                warn(`Attempting to mutate prop "${key}". Props are readonly.`, instance);
                return false;
            }
            if (key[0] === '$' && key.slice(1) in instance) {
                warn(`Attempting to mutate public property "${key}". ` +
                        `Properties starting with $ are reserved and readonly.`, instance);
                return false;
            }
            else {
                if (key in instance.appContext.config.globalProperties) {
                    Object.defineProperty(ctx, key, {
                        enumerable: true,
                        configurable: true,
                        value
                    });
                }
                else {
                    ctx[key] = value;
                }
            }
            return true;
        },
        has({ _: { data, setupState, accessCache, ctx, type, appContext } }, key) {
            let normalizedProps;
            return (accessCache[key] !== undefined ||
                (data !== EMPTY_OBJ && hasOwn(data, key)) ||
                (setupState !== EMPTY_OBJ && hasOwn(setupState, key)) ||
                ((normalizedProps = normalizePropsOptions(type)[0]) &&
                    hasOwn(normalizedProps, key)) ||
                hasOwn(ctx, key) ||
                hasOwn(publicPropertiesMap, key) ||
                hasOwn(appContext.config.globalProperties, key));
        }
    };
    {
        PublicInstanceProxyHandlers.ownKeys = (target) => {
            warn(`Avoid app logic that relies on enumerating keys on a component instance. ` +
                `The keys will be empty in production mode to avoid performance overhead.`);
            return Reflect.ownKeys(target);
        };
    }
    const RuntimeCompiledPublicInstanceProxyHandlers = extend({}, PublicInstanceProxyHandlers, {
        get(target, key) {
            // fast path for unscopables when using `with` block
            if (key === Symbol.unscopables) {
                return;
            }
            return PublicInstanceProxyHandlers.get(target, key, target);
        },
        has(_, key) {
            const has = key[0] !== '_' && !isGloballyWhitelisted(key);
            if (!has && PublicInstanceProxyHandlers.has(_, key)) {
                warn(`Property ${JSON.stringify(key)} should not start with _ which is a reserved prefix for Vue internals.`);
            }
            return has;
        }
    });
    let currentInstance = null;
    const getCurrentInstance = () => currentInstance || currentRenderingInstance;
    const setCurrentInstance = (instance) => {
        currentInstance = instance;
    };
    const isBuiltInTag = /*#__PURE__*/ makeMap('slot,component');
    let isInSSRComponentSetup = false;
    // record effects created during a component's setup() so that they can be
    // stopped when the component unmounts
    function recordInstanceBoundEffect(effect$$1) {
        if (currentInstance) {
            (currentInstance.effects || (currentInstance.effects = [])).push(effect$$1);
        }
    }
    const classifyRE = /(?:^|[-_])(\w)/g;
    const classify = (str) => str.replace(classifyRE, c => c.toUpperCase()).replace(/[-_]/g, '');
    function formatComponentName(instance, Component, isRoot = false) {
        let name = isFunction(Component)
            ? Component.displayName || Component.name
            : Component.name;
        if (!name && Component.__file) {
            const match = Component.__file.match(/([^/\\]+)\.vue$/);
            if (match) {
                name = match[1];
            }
        }
        if (!name && instance && instance.parent) {
            // try to infer the name based on local resolution
            const registry = instance.parent.components;
            for (const key in registry) {
                if (registry[key] === Component) {
                    name = key;
                    break;
                }
            }
        }
        return name ? classify(name) : isRoot ? `App` : `Anonymous`;
    }

    // Actual implementation
    function h(type, propsOrChildren, children) {
        if (arguments.length === 2) {
            if (isObject(propsOrChildren) && !isArray(propsOrChildren)) {
                // single vnode without props
                if (isVNode(propsOrChildren)) {
                    return createVNode(type, null, [propsOrChildren]);
                }
                // props without children
                return createVNode(type, propsOrChildren);
            }
            else {
                // omit props
                return createVNode(type, null, propsOrChildren);
            }
        }
        else {
            if (isVNode(children)) {
                children = [children];
            }
            return createVNode(type, propsOrChildren, children);
        }
    }
    /**
     * @private
     */
    const _camelize = camelize;

    const svgNS = 'http://www.w3.org/2000/svg';
    const doc = (typeof document !== 'undefined' ? document : null);
    let tempContainer;
    let tempSVGContainer;
    const nodeOps = {
        insert: (child, parent, anchor) => {
            parent.insertBefore(child, anchor || null);
        },
        remove: child => {
            const parent = child.parentNode;
            if (parent) {
                parent.removeChild(child);
            }
        },
        createElement: (tag, isSVG, is) => isSVG
            ? doc.createElementNS(svgNS, tag)
            : doc.createElement(tag, is ? { is } : undefined),
        createText: text => doc.createTextNode(text),
        createComment: text => doc.createComment(text),
        setText: (node, text) => {
            node.nodeValue = text;
        },
        setElementText: (el, text) => {
            el.textContent = text;
        },
        parentNode: node => node.parentNode,
        nextSibling: node => node.nextSibling,
        querySelector: selector => doc.querySelector(selector),
        setScopeId(el, id) {
            el.setAttribute(id, '');
        },
        cloneNode(el) {
            return el.cloneNode(true);
        },
        // __UNSAFE__
        // Reason: innerHTML.
        // Static content here can only come from compiled templates.
        // As long as the user only uses trusted templates, this is safe.
        insertStaticContent(content, parent, anchor, isSVG) {
            const temp = isSVG
                ? tempSVGContainer ||
                    (tempSVGContainer = doc.createElementNS(svgNS, 'svg'))
                : tempContainer || (tempContainer = doc.createElement('div'));
            temp.innerHTML = content;
            const first = temp.firstChild;
            let node = first;
            let last = node;
            while (node) {
                last = node;
                nodeOps.insert(node, parent, anchor);
                node = temp.firstChild;
            }
            return [first, last];
        }
    };

    // compiler should normalize class + :class bindings on the same element
    // into a single binding ['staticClass', dynamic]
    function patchClass(el, value, isSVG) {
        if (value == null) {
            value = '';
        }
        if (isSVG) {
            el.setAttribute('class', value);
        }
        else {
            // directly setting className should be faster than setAttribute in theory
            // if this is an element during a transition, take the temporary transition
            // classes into account.
            const transitionClasses = el._vtc;
            if (transitionClasses) {
                value = (value
                    ? [value, ...transitionClasses]
                    : [...transitionClasses]).join(' ');
            }
            el.className = value;
        }
    }

    function patchStyle(el, prev, next) {
        const style = el.style;
        if (!next) {
            el.removeAttribute('style');
        }
        else if (isString(next)) {
            if (prev !== next) {
                style.cssText = next;
            }
        }
        else {
            for (const key in next) {
                setStyle(style, key, next[key]);
            }
            if (prev && !isString(prev)) {
                for (const key in prev) {
                    if (next[key] == null) {
                        setStyle(style, key, '');
                    }
                }
            }
        }
    }
    const importantRE = /\s*!important$/;
    function setStyle(style, name, val) {
        if (name.startsWith('--')) {
            // custom property definition
            style.setProperty(name, val);
        }
        else {
            const prefixed = autoPrefix(style, name);
            if (importantRE.test(val)) {
                // !important
                style.setProperty(hyphenate(prefixed), val.replace(importantRE, ''), 'important');
            }
            else {
                style[prefixed] = val;
            }
        }
    }
    const prefixes = ['Webkit', 'Moz', 'ms'];
    const prefixCache = {};
    function autoPrefix(style, rawName) {
        const cached = prefixCache[rawName];
        if (cached) {
            return cached;
        }
        let name = _camelize(rawName);
        if (name !== 'filter' && name in style) {
            return (prefixCache[rawName] = name);
        }
        name = capitalize(name);
        for (let i = 0; i < prefixes.length; i++) {
            const prefixed = prefixes[i] + name;
            if (prefixed in style) {
                return (prefixCache[rawName] = prefixed);
            }
        }
        return rawName;
    }

    const xlinkNS = 'http://www.w3.org/1999/xlink';
    function patchAttr(el, key, value, isSVG) {
        if (isSVG && key.startsWith('xlink:')) {
            if (value == null) {
                el.removeAttributeNS(xlinkNS, key.slice(6, key.length));
            }
            else {
                el.setAttributeNS(xlinkNS, key, value);
            }
        }
        else {
            // note we are only checking boolean attributes that don't have a
            // corresponding dom prop of the same name here.
            const isBoolean = isSpecialBooleanAttr(key);
            if (value == null || (isBoolean && value === false)) {
                el.removeAttribute(key);
            }
            else {
                el.setAttribute(key, isBoolean ? '' : value);
            }
        }
    }

    // __UNSAFE__
    // functions. The user is responsible for using them with only trusted content.
    function patchDOMProp(el, key, value, 
    // the following args are passed only due to potential innerHTML/textContent
    // overriding existing VNodes, in which case the old tree must be properly
    // unmounted.
    prevChildren, parentComponent, parentSuspense, unmountChildren) {
        if (key === 'innerHTML' || key === 'textContent') {
            if (prevChildren) {
                unmountChildren(prevChildren, parentComponent, parentSuspense);
            }
            el[key] = value == null ? '' : value;
            return;
        }
        if (key === 'value' && el.tagName !== 'PROGRESS') {
            // store value as _value as well since
            // non-string values will be stringified.
            el._value = value;
            el.value = value == null ? '' : value;
            return;
        }
        if (value === '' && typeof el[key] === 'boolean') {
            // e.g. <select multiple> compiles to { multiple: '' }
            el[key] = true;
        }
        else if (value == null && typeof el[key] === 'string') {
            // e.g. <div :id="null">
            el[key] = '';
        }
        else {
            // some properties perform value validation and throw
            try {
                el[key] = value;
            }
            catch (e) {
                {
                    warn(`Failed setting prop "${key}" on <${el.tagName.toLowerCase()}>: ` +
                        `value ${value} is invalid.`, e);
                }
            }
        }
    }

    // Async edge case fix requires storing an event listener's attach timestamp.
    let _getNow = Date.now;
    // Determine what event timestamp the browser is using. Annoyingly, the
    // timestamp can either be hi-res (relative to page load) or low-res
    // (relative to UNIX epoch), so in order to compare time we have to use the
    // same timestamp type when saving the flush timestamp.
    if (typeof document !== 'undefined' &&
        _getNow() > document.createEvent('Event').timeStamp) {
        // if the low-res timestamp which is bigger than the event timestamp
        // (which is evaluated AFTER) it means the event is using a hi-res timestamp,
        // and we need to use the hi-res version for event listeners as well.
        _getNow = () => performance.now();
    }
    // To avoid the overhead of repeatedly calling performance.now(), we cache
    // and use the same timestamp for all event listeners attached in the same tick.
    let cachedNow = 0;
    const p$1 = Promise.resolve();
    const reset = () => {
        cachedNow = 0;
    };
    const getNow = () => cachedNow || (p$1.then(reset), (cachedNow = _getNow()));
    function addEventListener(el, event, handler, options) {
        el.addEventListener(event, handler, options);
    }
    function removeEventListener(el, event, handler, options) {
        el.removeEventListener(event, handler, options);
    }
    function patchEvent(el, rawName, prevValue, nextValue, instance = null) {
        const name = rawName.slice(2).toLowerCase();
        const prevOptions = prevValue && 'options' in prevValue && prevValue.options;
        const nextOptions = nextValue && 'options' in nextValue && nextValue.options;
        const invoker = prevValue && prevValue.invoker;
        const value = nextValue && 'handler' in nextValue ? nextValue.handler : nextValue;
        if (prevOptions || nextOptions) {
            const prev = prevOptions || EMPTY_OBJ;
            const next = nextOptions || EMPTY_OBJ;
            if (prev.capture !== next.capture ||
                prev.passive !== next.passive ||
                prev.once !== next.once) {
                if (invoker) {
                    removeEventListener(el, name, invoker, prev);
                }
                if (nextValue && value) {
                    const invoker = createInvoker(value, instance);
                    nextValue.invoker = invoker;
                    addEventListener(el, name, invoker, next);
                }
                return;
            }
        }
        if (nextValue && value) {
            if (invoker) {
                prevValue.invoker = null;
                invoker.value = value;
                nextValue.invoker = invoker;
                invoker.lastUpdated = getNow();
            }
            else {
                addEventListener(el, name, createInvoker(value, instance), nextOptions || void 0);
            }
        }
        else if (invoker) {
            removeEventListener(el, name, invoker, prevOptions || void 0);
        }
    }
    function createInvoker(initialValue, instance) {
        const invoker = (e) => {
            // async edge case #6566: inner click event triggers patch, event handler
            // attached to outer element during patch, and triggered again. This
            // happens because browsers fire microtask ticks between event propagation.
            // the solution is simple: we save the timestamp when a handler is attached,
            // and the handler would only fire if the event passed to it was fired
            // AFTER it was attached.
            const timeStamp = e.timeStamp || _getNow();
            if (timeStamp >= invoker.lastUpdated - 1) {
                callWithAsyncErrorHandling(patchStopImmediatePropagation(e, invoker.value), instance, 5 /* NATIVE_EVENT_HANDLER */, [e]);
            }
        };
        invoker.value = initialValue;
        initialValue.invoker = invoker;
        invoker.lastUpdated = getNow();
        return invoker;
    }
    function patchStopImmediatePropagation(e, value) {
        if (isArray(value)) {
            const originalStop = e.stopImmediatePropagation;
            e.stopImmediatePropagation = () => {
                originalStop.call(e);
                e._stopped = true;
            };
            return value.map(fn => (e) => !e._stopped && fn(e));
        }
        else {
            return value;
        }
    }

    const nativeOnRE = /^on[a-z]/;
    const forcePatchProp = (_, key) => key === 'value';
    const patchProp = (el, key, prevValue, nextValue, isSVG = false, prevChildren, parentComponent, parentSuspense, unmountChildren) => {
        switch (key) {
            // special
            case 'class':
                patchClass(el, nextValue, isSVG);
                break;
            case 'style':
                patchStyle(el, prevValue, nextValue);
                break;
            default:
                if (isOn(key)) {
                    // ignore v-model listeners
                    if (!key.startsWith('onUpdate:')) {
                        patchEvent(el, key, prevValue, nextValue, parentComponent);
                    }
                }
                else if (shouldSetAsProp(el, key, nextValue, isSVG)) {
                    patchDOMProp(el, key, nextValue, prevChildren, parentComponent, parentSuspense, unmountChildren);
                }
                else {
                    // special case for <input v-model type="checkbox"> with
                    // :true-value & :false-value
                    // store value as dom properties since non-string values will be
                    // stringified.
                    if (key === 'true-value') {
                        el._trueValue = nextValue;
                    }
                    else if (key === 'false-value') {
                        el._falseValue = nextValue;
                    }
                    patchAttr(el, key, nextValue, isSVG);
                }
                break;
        }
    };
    function shouldSetAsProp(el, key, value, isSVG) {
        if (isSVG) {
            // most keys must be set as attribute on svg elements to work
            // ...except innerHTML
            if (key === 'innerHTML') {
                return true;
            }
            // or native onclick with function values
            if (key in el && nativeOnRE.test(key) && isFunction(value)) {
                return true;
            }
            return false;
        }
        // spellcheck and draggable are numerated attrs, however their
        // corresponding DOM properties are actually booleans - this leads to
        // setting it with a string "false" value leading it to be coerced to
        // `true`, so we need to always treat them as attributes.
        // Note that `contentEditable` doesn't have this problem: its DOM
        // property is also enumerated string values.
        if (key === 'spellcheck' || key === 'draggable') {
            return false;
        }
        // #1526 <input list> must be set as attribute
        if (key === 'list' && el.tagName === 'INPUT') {
            return false;
        }
        // native onclick with string value, must be set as attribute
        if (nativeOnRE.test(key) && isString(value)) {
            return false;
        }
        return key in el;
    }

    const TRANSITION = 'transition';
    const ANIMATION = 'animation';
    // DOM Transition is a higher-order-component based on the platform-agnostic
    // base Transition component, with DOM-specific logic.
    const Transition = (props, { slots }) => h(BaseTransition, resolveTransitionProps(props), slots);
    Transition.displayName = 'Transition';
    const DOMTransitionPropsValidators = {
        name: String,
        type: String,
        css: {
            type: Boolean,
            default: true
        },
        duration: [String, Number, Object],
        enterFromClass: String,
        enterActiveClass: String,
        enterToClass: String,
        appearFromClass: String,
        appearActiveClass: String,
        appearToClass: String,
        leaveFromClass: String,
        leaveActiveClass: String,
        leaveToClass: String
    };
    const TransitionPropsValidators = (Transition.props = extend({}, BaseTransition.props, DOMTransitionPropsValidators));
    function resolveTransitionProps(rawProps) {
        let { name = 'v', type, css = true, duration, enterFromClass = `${name}-enter-from`, enterActiveClass = `${name}-enter-active`, enterToClass = `${name}-enter-to`, appearFromClass = enterFromClass, appearActiveClass = enterActiveClass, appearToClass = enterToClass, leaveFromClass = `${name}-leave-from`, leaveActiveClass = `${name}-leave-active`, leaveToClass = `${name}-leave-to` } = rawProps;
        const baseProps = {};
        for (const key in rawProps) {
            if (!(key in DOMTransitionPropsValidators)) {
                baseProps[key] = rawProps[key];
            }
        }
        if (!css) {
            return baseProps;
        }
        const durations = normalizeDuration(duration);
        const enterDuration = durations && durations[0];
        const leaveDuration = durations && durations[1];
        const { onBeforeEnter, onEnter, onEnterCancelled, onLeave, onLeaveCancelled, onBeforeAppear = onBeforeEnter, onAppear = onEnter, onAppearCancelled = onEnterCancelled } = baseProps;
        const finishEnter = (el, isAppear, done) => {
            removeTransitionClass(el, isAppear ? appearToClass : enterToClass);
            removeTransitionClass(el, isAppear ? appearActiveClass : enterActiveClass);
            done && done();
        };
        const finishLeave = (el, done) => {
            removeTransitionClass(el, leaveToClass);
            removeTransitionClass(el, leaveActiveClass);
            done && done();
        };
        const makeEnterHook = (isAppear) => {
            return (el, done) => {
                const hook = isAppear ? onAppear : onEnter;
                const resolve = () => finishEnter(el, isAppear, done);
                hook && hook(el, resolve);
                nextFrame(() => {
                    removeTransitionClass(el, isAppear ? appearFromClass : enterFromClass);
                    addTransitionClass(el, isAppear ? appearToClass : enterToClass);
                    if (!(hook && hook.length > 1)) {
                        if (enterDuration) {
                            setTimeout(resolve, enterDuration);
                        }
                        else {
                            whenTransitionEnds(el, type, resolve);
                        }
                    }
                });
            };
        };
        return extend(baseProps, {
            onBeforeEnter(el) {
                onBeforeEnter && onBeforeEnter(el);
                addTransitionClass(el, enterActiveClass);
                addTransitionClass(el, enterFromClass);
            },
            onBeforeAppear(el) {
                onBeforeAppear && onBeforeAppear(el);
                addTransitionClass(el, appearActiveClass);
                addTransitionClass(el, appearFromClass);
            },
            onEnter: makeEnterHook(false),
            onAppear: makeEnterHook(true),
            onLeave(el, done) {
                const resolve = () => finishLeave(el, done);
                addTransitionClass(el, leaveActiveClass);
                addTransitionClass(el, leaveFromClass);
                nextFrame(() => {
                    removeTransitionClass(el, leaveFromClass);
                    addTransitionClass(el, leaveToClass);
                    if (!(onLeave && onLeave.length > 1)) {
                        if (leaveDuration) {
                            setTimeout(resolve, leaveDuration);
                        }
                        else {
                            whenTransitionEnds(el, type, resolve);
                        }
                    }
                });
                onLeave && onLeave(el, resolve);
            },
            onEnterCancelled(el) {
                finishEnter(el, false);
                onEnterCancelled && onEnterCancelled(el);
            },
            onAppearCancelled(el) {
                finishEnter(el, true);
                onAppearCancelled && onAppearCancelled(el);
            },
            onLeaveCancelled(el) {
                finishLeave(el);
                onLeaveCancelled && onLeaveCancelled(el);
            }
        });
    }
    function normalizeDuration(duration) {
        if (duration == null) {
            return null;
        }
        else if (isObject(duration)) {
            return [NumberOf(duration.enter), NumberOf(duration.leave)];
        }
        else {
            const n = NumberOf(duration);
            return [n, n];
        }
    }
    function NumberOf(val) {
        const res = toNumber(val);
        validateDuration(res);
        return res;
    }
    function validateDuration(val) {
        if (typeof val !== 'number') {
            warn(`<transition> explicit duration is not a valid number - ` +
                `got ${JSON.stringify(val)}.`);
        }
        else if (isNaN(val)) {
            warn(`<transition> explicit duration is NaN - ` +
                'the duration expression might be incorrect.');
        }
    }
    function addTransitionClass(el, cls) {
        cls.split(/\s+/).forEach(c => c && el.classList.add(c));
        (el._vtc ||
            (el._vtc = new Set())).add(cls);
    }
    function removeTransitionClass(el, cls) {
        cls.split(/\s+/).forEach(c => c && el.classList.remove(c));
        const { _vtc } = el;
        if (_vtc) {
            _vtc.delete(cls);
            if (!_vtc.size) {
                el._vtc = undefined;
            }
        }
    }
    function nextFrame(cb) {
        requestAnimationFrame(() => {
            requestAnimationFrame(cb);
        });
    }
    function whenTransitionEnds(el, expectedType, cb) {
        const { type, timeout, propCount } = getTransitionInfo(el, expectedType);
        if (!type) {
            return cb();
        }
        const endEvent = type + 'end';
        let ended = 0;
        const end = () => {
            el.removeEventListener(endEvent, onEnd);
            cb();
        };
        const onEnd = (e) => {
            if (e.target === el) {
                if (++ended >= propCount) {
                    end();
                }
            }
        };
        setTimeout(() => {
            if (ended < propCount) {
                end();
            }
        }, timeout + 1);
        el.addEventListener(endEvent, onEnd);
    }
    function getTransitionInfo(el, expectedType) {
        const styles = window.getComputedStyle(el);
        // JSDOM may return undefined for transition properties
        const getStyleProperties = (key) => (styles[key] || '').split(', ');
        const transitionDelays = getStyleProperties(TRANSITION + 'Delay');
        const transitionDurations = getStyleProperties(TRANSITION + 'Duration');
        const transitionTimeout = getTimeout(transitionDelays, transitionDurations);
        const animationDelays = getStyleProperties(ANIMATION + 'Delay');
        const animationDurations = getStyleProperties(ANIMATION + 'Duration');
        const animationTimeout = getTimeout(animationDelays, animationDurations);
        let type = null;
        let timeout = 0;
        let propCount = 0;
        /* istanbul ignore if */
        if (expectedType === TRANSITION) {
            if (transitionTimeout > 0) {
                type = TRANSITION;
                timeout = transitionTimeout;
                propCount = transitionDurations.length;
            }
        }
        else if (expectedType === ANIMATION) {
            if (animationTimeout > 0) {
                type = ANIMATION;
                timeout = animationTimeout;
                propCount = animationDurations.length;
            }
        }
        else {
            timeout = Math.max(transitionTimeout, animationTimeout);
            type =
                timeout > 0
                    ? transitionTimeout > animationTimeout
                        ? TRANSITION
                        : ANIMATION
                    : null;
            propCount = type
                ? type === TRANSITION
                    ? transitionDurations.length
                    : animationDurations.length
                : 0;
        }
        const hasTransform = type === TRANSITION &&
            /\b(transform|all)(,|$)/.test(styles[TRANSITION + 'Property']);
        return {
            type,
            timeout,
            propCount,
            hasTransform
        };
    }
    function getTimeout(delays, durations) {
        while (delays.length < durations.length) {
            delays = delays.concat(delays);
        }
        return Math.max(...durations.map((d, i) => toMs(d) + toMs(delays[i])));
    }
    // Old versions of Chromium (below 61.0.3163.100) formats floating pointer
    // numbers in a locale-dependent way, using a comma instead of a dot.
    // If comma is not replaced with a dot, the input will be rounded down
    // (i.e. acting as a floor function) causing unexpected behaviors
    function toMs(s) {
        return Number(s.slice(0, -1).replace(',', '.')) * 1000;
    }

    function toRaw$1(observed) {
        return ((observed && toRaw$1(observed["__v_raw" /* RAW */])) || observed);
    }

    const positionMap = new WeakMap();
    const newPositionMap = new WeakMap();
    const TransitionGroupImpl = {
        name: 'TransitionGroup',
        props: extend({}, TransitionPropsValidators, {
            tag: String,
            moveClass: String
        }),
        setup(props, { slots }) {
            const instance = getCurrentInstance();
            const state = useTransitionState();
            let prevChildren;
            let children;
            onUpdated(() => {
                // children is guaranteed to exist after initial render
                if (!prevChildren.length) {
                    return;
                }
                const moveClass = props.moveClass || `${props.name || 'v'}-move`;
                if (!hasCSSTransform(prevChildren[0].el, instance.vnode.el, moveClass)) {
                    return;
                }
                // we divide the work into three loops to avoid mixing DOM reads and writes
                // in each iteration - which helps prevent layout thrashing.
                prevChildren.forEach(callPendingCbs);
                prevChildren.forEach(recordPosition);
                const movedChildren = prevChildren.filter(applyTranslation);
                // force reflow to put everything in position
                forceReflow();
                movedChildren.forEach(c => {
                    const el = c.el;
                    const style = el.style;
                    addTransitionClass(el, moveClass);
                    style.transform = style.webkitTransform = style.transitionDuration = '';
                    const cb = (el._moveCb = (e) => {
                        if (e && e.target !== el) {
                            return;
                        }
                        if (!e || /transform$/.test(e.propertyName)) {
                            el.removeEventListener('transitionend', cb);
                            el._moveCb = null;
                            removeTransitionClass(el, moveClass);
                        }
                    });
                    el.addEventListener('transitionend', cb);
                });
            });
            return () => {
                const rawProps = toRaw$1(props);
                const cssTransitionProps = resolveTransitionProps(rawProps);
                const tag = rawProps.tag || Fragment;
                prevChildren = children;
                children = slots.default ? getTransitionRawChildren(slots.default()) : [];
                for (let i = 0; i < children.length; i++) {
                    const child = children[i];
                    if (child.key != null) {
                        setTransitionHooks(child, resolveTransitionHooks(child, cssTransitionProps, state, instance));
                    }
                    else {
                        warn(`<TransitionGroup> children must be keyed.`);
                    }
                }
                if (prevChildren) {
                    for (let i = 0; i < prevChildren.length; i++) {
                        const child = prevChildren[i];
                        setTransitionHooks(child, resolveTransitionHooks(child, cssTransitionProps, state, instance));
                        positionMap.set(child, child.el.getBoundingClientRect());
                    }
                }
                return createVNode(tag, null, children);
            };
        }
    };
    // remove mode props as TransitionGroup doesn't support it
    delete TransitionGroupImpl.props.mode;
    function callPendingCbs(c) {
        const el = c.el;
        if (el._moveCb) {
            el._moveCb();
        }
        if (el._enterCb) {
            el._enterCb();
        }
    }
    function recordPosition(c) {
        newPositionMap.set(c, c.el.getBoundingClientRect());
    }
    function applyTranslation(c) {
        const oldPos = positionMap.get(c);
        const newPos = newPositionMap.get(c);
        const dx = oldPos.left - newPos.left;
        const dy = oldPos.top - newPos.top;
        if (dx || dy) {
            const s = c.el.style;
            s.transform = s.webkitTransform = `translate(${dx}px,${dy}px)`;
            s.transitionDuration = '0s';
            return c;
        }
    }
    // this is put in a dedicated function to avoid the line from being treeshaken
    function forceReflow() {
        return document.body.offsetHeight;
    }
    function hasCSSTransform(el, root, moveClass) {
        // Detect whether an element with the move class applied has
        // CSS transitions. Since the element may be inside an entering
        // transition at this very moment, we make a clone of it and remove
        // all other transition classes applied to ensure only the move class
        // is applied.
        const clone = el.cloneNode();
        if (el._vtc) {
            el._vtc.forEach(cls => {
                cls.split(/\s+/).forEach(c => c && clone.classList.remove(c));
            });
        }
        moveClass.split(/\s+/).forEach(c => c && clone.classList.add(c));
        clone.style.display = 'none';
        const container = (root.nodeType === 1
            ? root
            : root.parentNode);
        container.appendChild(clone);
        const { hasTransform } = getTransitionInfo(clone);
        container.removeChild(clone);
        return hasTransform;
    }

    const rendererOptions = extend({ patchProp, forcePatchProp }, nodeOps);

    var ActivityIcon = {
      name: 'ActivityIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-activity"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "22 12 18 12 15 21 9 3 6 12 2 12"
        })]); };
      }

    };

    var AirplayIcon = {
      name: 'AirplayIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-airplay"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M5 17H4a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h16a2 2 0 0 1 2 2v10a2 2 0 0 1-2 2h-1"
        }), createVNode("polygon", {
          "points": "12 15 17 21 7 21 12 15"
        })]); };
      }

    };

    var AlertCircleIcon = {
      name: 'AlertCircleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-alert-circle"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("line", {
          "x1": "12",
          "y1": "8",
          "x2": "12",
          "y2": "12"
        }), createVNode("line", {
          "x1": "12",
          "y1": "16",
          "x2": "12.01",
          "y2": "16"
        })]); };
      }

    };

    var AlertOctagonIcon = {
      name: 'AlertOctagonIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-alert-octagon"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "7.86 2 16.14 2 22 7.86 22 16.14 16.14 22 7.86 22 2 16.14 2 7.86 7.86 2"
        }), createVNode("line", {
          "x1": "12",
          "y1": "8",
          "x2": "12",
          "y2": "12"
        }), createVNode("line", {
          "x1": "12",
          "y1": "16",
          "x2": "12.01",
          "y2": "16"
        })]); };
      }

    };

    var AlertTriangleIcon = {
      name: 'AlertTriangleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-alert-triangle"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"
        }), createVNode("line", {
          "x1": "12",
          "y1": "9",
          "x2": "12",
          "y2": "13"
        }), createVNode("line", {
          "x1": "12",
          "y1": "17",
          "x2": "12.01",
          "y2": "17"
        })]); };
      }

    };

    var AlignCenterIcon = {
      name: 'AlignCenterIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-align-center"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "18",
          "y1": "10",
          "x2": "6",
          "y2": "10"
        }), createVNode("line", {
          "x1": "21",
          "y1": "6",
          "x2": "3",
          "y2": "6"
        }), createVNode("line", {
          "x1": "21",
          "y1": "14",
          "x2": "3",
          "y2": "14"
        }), createVNode("line", {
          "x1": "18",
          "y1": "18",
          "x2": "6",
          "y2": "18"
        })]); };
      }

    };

    var AlignJustifyIcon = {
      name: 'AlignJustifyIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-align-justify"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "21",
          "y1": "10",
          "x2": "3",
          "y2": "10"
        }), createVNode("line", {
          "x1": "21",
          "y1": "6",
          "x2": "3",
          "y2": "6"
        }), createVNode("line", {
          "x1": "21",
          "y1": "14",
          "x2": "3",
          "y2": "14"
        }), createVNode("line", {
          "x1": "21",
          "y1": "18",
          "x2": "3",
          "y2": "18"
        })]); };
      }

    };

    var AlignLeftIcon = {
      name: 'AlignLeftIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-align-left"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "17",
          "y1": "10",
          "x2": "3",
          "y2": "10"
        }), createVNode("line", {
          "x1": "21",
          "y1": "6",
          "x2": "3",
          "y2": "6"
        }), createVNode("line", {
          "x1": "21",
          "y1": "14",
          "x2": "3",
          "y2": "14"
        }), createVNode("line", {
          "x1": "17",
          "y1": "18",
          "x2": "3",
          "y2": "18"
        })]); };
      }

    };

    var AlignRightIcon = {
      name: 'AlignRightIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-align-right"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "21",
          "y1": "10",
          "x2": "7",
          "y2": "10"
        }), createVNode("line", {
          "x1": "21",
          "y1": "6",
          "x2": "3",
          "y2": "6"
        }), createVNode("line", {
          "x1": "21",
          "y1": "14",
          "x2": "3",
          "y2": "14"
        }), createVNode("line", {
          "x1": "21",
          "y1": "18",
          "x2": "7",
          "y2": "18"
        })]); };
      }

    };

    var AnchorIcon = {
      name: 'AnchorIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-anchor"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "5",
          "r": "3"
        }), createVNode("line", {
          "x1": "12",
          "y1": "22",
          "x2": "12",
          "y2": "8"
        }), createVNode("path", {
          "d": "M5 12H2a10 10 0 0 0 20 0h-3"
        })]); };
      }

    };

    var ApertureIcon = {
      name: 'ApertureIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-aperture"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("line", {
          "x1": "14.31",
          "y1": "8",
          "x2": "20.05",
          "y2": "17.94"
        }), createVNode("line", {
          "x1": "9.69",
          "y1": "8",
          "x2": "21.17",
          "y2": "8"
        }), createVNode("line", {
          "x1": "7.38",
          "y1": "12",
          "x2": "13.12",
          "y2": "2.06"
        }), createVNode("line", {
          "x1": "9.69",
          "y1": "16",
          "x2": "3.95",
          "y2": "6.06"
        }), createVNode("line", {
          "x1": "14.31",
          "y1": "16",
          "x2": "2.83",
          "y2": "16"
        }), createVNode("line", {
          "x1": "16.62",
          "y1": "12",
          "x2": "10.88",
          "y2": "21.94"
        })]); };
      }

    };

    var ArchiveIcon = {
      name: 'ArchiveIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-archive"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "21 8 21 21 3 21 3 8"
        }), createVNode("rect", {
          "x": "1",
          "y": "3",
          "width": "22",
          "height": "5"
        }), createVNode("line", {
          "x1": "10",
          "y1": "12",
          "x2": "14",
          "y2": "12"
        })]); };
      }

    };

    var ArrowDownCircleIcon = {
      name: 'ArrowDownCircleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-arrow-down-circle"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("polyline", {
          "points": "8 12 12 16 16 12"
        }), createVNode("line", {
          "x1": "12",
          "y1": "8",
          "x2": "12",
          "y2": "16"
        })]); };
      }

    };

    var ArrowDownLeftIcon = {
      name: 'ArrowDownLeftIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-arrow-down-left"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "17",
          "y1": "7",
          "x2": "7",
          "y2": "17"
        }), createVNode("polyline", {
          "points": "17 17 7 17 7 7"
        })]); };
      }

    };

    var ArrowDownRightIcon = {
      name: 'ArrowDownRightIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-arrow-down-right"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "7",
          "y1": "7",
          "x2": "17",
          "y2": "17"
        }), createVNode("polyline", {
          "points": "17 7 17 17 7 17"
        })]); };
      }

    };

    var ArrowDownIcon = {
      name: 'ArrowDownIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-arrow-down"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "12",
          "y1": "5",
          "x2": "12",
          "y2": "19"
        }), createVNode("polyline", {
          "points": "19 12 12 19 5 12"
        })]); };
      }

    };

    var ArrowLeftCircleIcon = {
      name: 'ArrowLeftCircleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-arrow-left-circle"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("polyline", {
          "points": "12 8 8 12 12 16"
        }), createVNode("line", {
          "x1": "16",
          "y1": "12",
          "x2": "8",
          "y2": "12"
        })]); };
      }

    };

    var ArrowLeftIcon = {
      name: 'ArrowLeftIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-arrow-left"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "19",
          "y1": "12",
          "x2": "5",
          "y2": "12"
        }), createVNode("polyline", {
          "points": "12 19 5 12 12 5"
        })]); };
      }

    };

    var ArrowRightCircleIcon = {
      name: 'ArrowRightCircleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-arrow-right-circle"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("polyline", {
          "points": "12 16 16 12 12 8"
        }), createVNode("line", {
          "x1": "8",
          "y1": "12",
          "x2": "16",
          "y2": "12"
        })]); };
      }

    };

    var ArrowRightIcon = {
      name: 'ArrowRightIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-arrow-right"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "5",
          "y1": "12",
          "x2": "19",
          "y2": "12"
        }), createVNode("polyline", {
          "points": "12 5 19 12 12 19"
        })]); };
      }

    };

    var ArrowUpCircleIcon = {
      name: 'ArrowUpCircleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-arrow-up-circle"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("polyline", {
          "points": "16 12 12 8 8 12"
        }), createVNode("line", {
          "x1": "12",
          "y1": "16",
          "x2": "12",
          "y2": "8"
        })]); };
      }

    };

    var ArrowUpLeftIcon = {
      name: 'ArrowUpLeftIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-arrow-up-left"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "17",
          "y1": "17",
          "x2": "7",
          "y2": "7"
        }), createVNode("polyline", {
          "points": "7 17 7 7 17 7"
        })]); };
      }

    };

    var ArrowUpRightIcon = {
      name: 'ArrowUpRightIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-arrow-up-right"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "7",
          "y1": "17",
          "x2": "17",
          "y2": "7"
        }), createVNode("polyline", {
          "points": "7 7 17 7 17 17"
        })]); };
      }

    };

    var ArrowUpIcon = {
      name: 'ArrowUpIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-arrow-up"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "12",
          "y1": "19",
          "x2": "12",
          "y2": "5"
        }), createVNode("polyline", {
          "points": "5 12 12 5 19 12"
        })]); };
      }

    };

    var AtSignIcon = {
      name: 'AtSignIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-at-sign"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "4"
        }), createVNode("path", {
          "d": "M16 8v5a3 3 0 0 0 6 0v-1a10 10 0 1 0-3.92 7.94"
        })]); };
      }

    };

    var AwardIcon = {
      name: 'AwardIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-award"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "8",
          "r": "7"
        }), createVNode("polyline", {
          "points": "8.21 13.89 7 23 12 20 17 23 15.79 13.88"
        })]); };
      }

    };

    var BarChart2Icon = {
      name: 'BarChart2Icon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-bar-chart-2"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "18",
          "y1": "20",
          "x2": "18",
          "y2": "10"
        }), createVNode("line", {
          "x1": "12",
          "y1": "20",
          "x2": "12",
          "y2": "4"
        }), createVNode("line", {
          "x1": "6",
          "y1": "20",
          "x2": "6",
          "y2": "14"
        })]); };
      }

    };

    var BarChartIcon = {
      name: 'BarChartIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-bar-chart"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "12",
          "y1": "20",
          "x2": "12",
          "y2": "10"
        }), createVNode("line", {
          "x1": "18",
          "y1": "20",
          "x2": "18",
          "y2": "4"
        }), createVNode("line", {
          "x1": "6",
          "y1": "20",
          "x2": "6",
          "y2": "16"
        })]); };
      }

    };

    var BatteryChargingIcon = {
      name: 'BatteryChargingIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-battery-charging"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M5 18H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h3.19M15 6h2a2 2 0 0 1 2 2v8a2 2 0 0 1-2 2h-3.19"
        }), createVNode("line", {
          "x1": "23",
          "y1": "13",
          "x2": "23",
          "y2": "11"
        }), createVNode("polyline", {
          "points": "11 6 7 12 13 12 9 18"
        })]); };
      }

    };

    var BatteryIcon = {
      name: 'BatteryIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-battery"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "1",
          "y": "6",
          "width": "18",
          "height": "12",
          "rx": "2",
          "ry": "2"
        }), createVNode("line", {
          "x1": "23",
          "y1": "13",
          "x2": "23",
          "y2": "11"
        })]); };
      }

    };

    var BellOffIcon = {
      name: 'BellOffIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-bell-off"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M13.73 21a2 2 0 0 1-3.46 0"
        }), createVNode("path", {
          "d": "M18.63 13A17.89 17.89 0 0 1 18 8"
        }), createVNode("path", {
          "d": "M6.26 6.26A5.86 5.86 0 0 0 6 8c0 7-3 9-3 9h14"
        }), createVNode("path", {
          "d": "M18 8a6 6 0 0 0-9.33-5"
        }), createVNode("line", {
          "x1": "1",
          "y1": "1",
          "x2": "23",
          "y2": "23"
        })]); };
      }

    };

    var BellIcon = {
      name: 'BellIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-bell"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M18 8A6 6 0 0 0 6 8c0 7-3 9-3 9h18s-3-2-3-9"
        }), createVNode("path", {
          "d": "M13.73 21a2 2 0 0 1-3.46 0"
        })]); };
      }

    };

    var BluetoothIcon = {
      name: 'BluetoothIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-bluetooth"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "6.5 6.5 17.5 17.5 12 23 12 1 17.5 6.5 6.5 17.5"
        })]); };
      }

    };

    var BoldIcon = {
      name: 'BoldIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-bold"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M6 4h8a4 4 0 0 1 4 4 4 4 0 0 1-4 4H6z"
        }), createVNode("path", {
          "d": "M6 12h9a4 4 0 0 1 4 4 4 4 0 0 1-4 4H6z"
        })]); };
      }

    };

    var BookOpenIcon = {
      name: 'BookOpenIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-book-open"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M2 3h6a4 4 0 0 1 4 4v14a3 3 0 0 0-3-3H2z"
        }), createVNode("path", {
          "d": "M22 3h-6a4 4 0 0 0-4 4v14a3 3 0 0 1 3-3h7z"
        })]); };
      }

    };

    var BookIcon = {
      name: 'BookIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-book"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M4 19.5A2.5 2.5 0 0 1 6.5 17H20"
        }), createVNode("path", {
          "d": "M6.5 2H20v20H6.5A2.5 2.5 0 0 1 4 19.5v-15A2.5 2.5 0 0 1 6.5 2z"
        })]); };
      }

    };

    var BookmarkIcon = {
      name: 'BookmarkIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-bookmark"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M19 21l-7-5-7 5V5a2 2 0 0 1 2-2h10a2 2 0 0 1 2 2z"
        })]); };
      }

    };

    var BoxIcon = {
      name: 'BoxIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-box"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"
        }), createVNode("polyline", {
          "points": "3.27 6.96 12 12.01 20.73 6.96"
        }), createVNode("line", {
          "x1": "12",
          "y1": "22.08",
          "x2": "12",
          "y2": "12"
        })]); };
      }

    };

    var BriefcaseIcon = {
      name: 'BriefcaseIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-briefcase"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "2",
          "y": "7",
          "width": "20",
          "height": "14",
          "rx": "2",
          "ry": "2"
        }), createVNode("path", {
          "d": "M16 21V5a2 2 0 0 0-2-2h-4a2 2 0 0 0-2 2v16"
        })]); };
      }

    };

    var CalendarIcon = {
      name: 'CalendarIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-calendar"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "3",
          "y": "4",
          "width": "18",
          "height": "18",
          "rx": "2",
          "ry": "2"
        }), createVNode("line", {
          "x1": "16",
          "y1": "2",
          "x2": "16",
          "y2": "6"
        }), createVNode("line", {
          "x1": "8",
          "y1": "2",
          "x2": "8",
          "y2": "6"
        }), createVNode("line", {
          "x1": "3",
          "y1": "10",
          "x2": "21",
          "y2": "10"
        })]); };
      }

    };

    var CameraOffIcon = {
      name: 'CameraOffIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-camera-off"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "1",
          "y1": "1",
          "x2": "23",
          "y2": "23"
        }), createVNode("path", {
          "d": "M21 21H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h3m3-3h6l2 3h4a2 2 0 0 1 2 2v9.34m-7.72-2.06a4 4 0 1 1-5.56-5.56"
        })]); };
      }

    };

    var CameraIcon = {
      name: 'CameraIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-camera"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"
        }), createVNode("circle", {
          "cx": "12",
          "cy": "13",
          "r": "4"
        })]); };
      }

    };

    var CastIcon = {
      name: 'CastIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-cast"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M2 16.1A5 5 0 0 1 5.9 20M2 12.05A9 9 0 0 1 9.95 20M2 8V6a2 2 0 0 1 2-2h16a2 2 0 0 1 2 2v12a2 2 0 0 1-2 2h-6"
        }), createVNode("line", {
          "x1": "2",
          "y1": "20",
          "x2": "2.01",
          "y2": "20"
        })]); };
      }

    };

    var CheckCircleIcon = {
      name: 'CheckCircleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-check-circle"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M22 11.08V12a10 10 0 1 1-5.93-9.14"
        }), createVNode("polyline", {
          "points": "22 4 12 14.01 9 11.01"
        })]); };
      }

    };

    var CheckSquareIcon = {
      name: 'CheckSquareIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-check-square"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "9 11 12 14 22 4"
        }), createVNode("path", {
          "d": "M21 12v7a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11"
        })]); };
      }

    };

    var CheckIcon = {
      name: 'CheckIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-check"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "20 6 9 17 4 12"
        })]); };
      }

    };

    var ChevronDownIcon = {
      name: 'ChevronDownIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-chevron-down"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "6 9 12 15 18 9"
        })]); };
      }

    };

    var ChevronLeftIcon = {
      name: 'ChevronLeftIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-chevron-left"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "15 18 9 12 15 6"
        })]); };
      }

    };

    var ChevronRightIcon = {
      name: 'ChevronRightIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-chevron-right"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "9 18 15 12 9 6"
        })]); };
      }

    };

    var ChevronUpIcon = {
      name: 'ChevronUpIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-chevron-up"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "18 15 12 9 6 15"
        })]); };
      }

    };

    var ChevronsDownIcon = {
      name: 'ChevronsDownIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-chevrons-down"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "7 13 12 18 17 13"
        }), createVNode("polyline", {
          "points": "7 6 12 11 17 6"
        })]); };
      }

    };

    var ChevronsLeftIcon = {
      name: 'ChevronsLeftIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-chevrons-left"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "11 17 6 12 11 7"
        }), createVNode("polyline", {
          "points": "18 17 13 12 18 7"
        })]); };
      }

    };

    var ChevronsRightIcon = {
      name: 'ChevronsRightIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-chevrons-right"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "13 17 18 12 13 7"
        }), createVNode("polyline", {
          "points": "6 17 11 12 6 7"
        })]); };
      }

    };

    var ChevronsUpIcon = {
      name: 'ChevronsUpIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-chevrons-up"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "17 11 12 6 7 11"
        }), createVNode("polyline", {
          "points": "17 18 12 13 7 18"
        })]); };
      }

    };

    var ChromeIcon = {
      name: 'ChromeIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-chrome"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "4"
        }), createVNode("line", {
          "x1": "21.17",
          "y1": "8",
          "x2": "12",
          "y2": "8"
        }), createVNode("line", {
          "x1": "3.95",
          "y1": "6.06",
          "x2": "8.54",
          "y2": "14"
        }), createVNode("line", {
          "x1": "10.88",
          "y1": "21.94",
          "x2": "15.46",
          "y2": "14"
        })]); };
      }

    };

    var CircleIcon = {
      name: 'CircleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-circle"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        })]); };
      }

    };

    var ClipboardIcon = {
      name: 'ClipboardIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-clipboard"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M16 4h2a2 2 0 0 1 2 2v14a2 2 0 0 1-2 2H6a2 2 0 0 1-2-2V6a2 2 0 0 1 2-2h2"
        }), createVNode("rect", {
          "x": "8",
          "y": "2",
          "width": "8",
          "height": "4",
          "rx": "1",
          "ry": "1"
        })]); };
      }

    };

    var ClockIcon = {
      name: 'ClockIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-clock"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("polyline", {
          "points": "12 6 12 12 16 14"
        })]); };
      }

    };

    var CloudDrizzleIcon = {
      name: 'CloudDrizzleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-cloud-drizzle"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "8",
          "y1": "19",
          "x2": "8",
          "y2": "21"
        }), createVNode("line", {
          "x1": "8",
          "y1": "13",
          "x2": "8",
          "y2": "15"
        }), createVNode("line", {
          "x1": "16",
          "y1": "19",
          "x2": "16",
          "y2": "21"
        }), createVNode("line", {
          "x1": "16",
          "y1": "13",
          "x2": "16",
          "y2": "15"
        }), createVNode("line", {
          "x1": "12",
          "y1": "21",
          "x2": "12",
          "y2": "23"
        }), createVNode("line", {
          "x1": "12",
          "y1": "15",
          "x2": "12",
          "y2": "17"
        }), createVNode("path", {
          "d": "M20 16.58A5 5 0 0 0 18 7h-1.26A8 8 0 1 0 4 15.25"
        })]); };
      }

    };

    var CloudLightningIcon = {
      name: 'CloudLightningIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-cloud-lightning"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M19 16.9A5 5 0 0 0 18 7h-1.26a8 8 0 1 0-11.62 9"
        }), createVNode("polyline", {
          "points": "13 11 9 17 15 17 11 23"
        })]); };
      }

    };

    var CloudOffIcon = {
      name: 'CloudOffIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-cloud-off"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M22.61 16.95A5 5 0 0 0 18 10h-1.26a8 8 0 0 0-7.05-6M5 5a8 8 0 0 0 4 15h9a5 5 0 0 0 1.7-.3"
        }), createVNode("line", {
          "x1": "1",
          "y1": "1",
          "x2": "23",
          "y2": "23"
        })]); };
      }

    };

    var CloudRainIcon = {
      name: 'CloudRainIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-cloud-rain"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "16",
          "y1": "13",
          "x2": "16",
          "y2": "21"
        }), createVNode("line", {
          "x1": "8",
          "y1": "13",
          "x2": "8",
          "y2": "21"
        }), createVNode("line", {
          "x1": "12",
          "y1": "15",
          "x2": "12",
          "y2": "23"
        }), createVNode("path", {
          "d": "M20 16.58A5 5 0 0 0 18 7h-1.26A8 8 0 1 0 4 15.25"
        })]); };
      }

    };

    var CloudSnowIcon = {
      name: 'CloudSnowIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-cloud-snow"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M20 17.58A5 5 0 0 0 18 8h-1.26A8 8 0 1 0 4 16.25"
        }), createVNode("line", {
          "x1": "8",
          "y1": "16",
          "x2": "8.01",
          "y2": "16"
        }), createVNode("line", {
          "x1": "8",
          "y1": "20",
          "x2": "8.01",
          "y2": "20"
        }), createVNode("line", {
          "x1": "12",
          "y1": "18",
          "x2": "12.01",
          "y2": "18"
        }), createVNode("line", {
          "x1": "12",
          "y1": "22",
          "x2": "12.01",
          "y2": "22"
        }), createVNode("line", {
          "x1": "16",
          "y1": "16",
          "x2": "16.01",
          "y2": "16"
        }), createVNode("line", {
          "x1": "16",
          "y1": "20",
          "x2": "16.01",
          "y2": "20"
        })]); };
      }

    };

    var CloudIcon = {
      name: 'CloudIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-cloud"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M18 10h-1.26A8 8 0 1 0 9 20h9a5 5 0 0 0 0-10z"
        })]); };
      }

    };

    var CodeIcon = {
      name: 'CodeIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-code"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "16 18 22 12 16 6"
        }), createVNode("polyline", {
          "points": "8 6 2 12 8 18"
        })]); };
      }

    };

    var CodepenIcon = {
      name: 'CodepenIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-codepen"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "12 2 22 8.5 22 15.5 12 22 2 15.5 2 8.5 12 2"
        }), createVNode("line", {
          "x1": "12",
          "y1": "22",
          "x2": "12",
          "y2": "15.5"
        }), createVNode("polyline", {
          "points": "22 8.5 12 15.5 2 8.5"
        }), createVNode("polyline", {
          "points": "2 15.5 12 8.5 22 15.5"
        }), createVNode("line", {
          "x1": "12",
          "y1": "2",
          "x2": "12",
          "y2": "8.5"
        })]); };
      }

    };

    var CodesandboxIcon = {
      name: 'CodesandboxIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-codesandbox"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"
        }), createVNode("polyline", {
          "points": "7.5 4.21 12 6.81 16.5 4.21"
        }), createVNode("polyline", {
          "points": "7.5 19.79 7.5 14.6 3 12"
        }), createVNode("polyline", {
          "points": "21 12 16.5 14.6 16.5 19.79"
        }), createVNode("polyline", {
          "points": "3.27 6.96 12 12.01 20.73 6.96"
        }), createVNode("line", {
          "x1": "12",
          "y1": "22.08",
          "x2": "12",
          "y2": "12"
        })]); };
      }

    };

    var CoffeeIcon = {
      name: 'CoffeeIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-coffee"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M18 8h1a4 4 0 0 1 0 8h-1"
        }), createVNode("path", {
          "d": "M2 8h16v9a4 4 0 0 1-4 4H6a4 4 0 0 1-4-4V8z"
        }), createVNode("line", {
          "x1": "6",
          "y1": "1",
          "x2": "6",
          "y2": "4"
        }), createVNode("line", {
          "x1": "10",
          "y1": "1",
          "x2": "10",
          "y2": "4"
        }), createVNode("line", {
          "x1": "14",
          "y1": "1",
          "x2": "14",
          "y2": "4"
        })]); };
      }

    };

    var ColumnsIcon = {
      name: 'ColumnsIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-columns"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M12 3h7a2 2 0 0 1 2 2v14a2 2 0 0 1-2 2h-7m0-18H5a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h7m0-18v18"
        })]); };
      }

    };

    var CommandIcon = {
      name: 'CommandIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-command"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M18 3a3 3 0 0 0-3 3v12a3 3 0 0 0 3 3 3 3 0 0 0 3-3 3 3 0 0 0-3-3H6a3 3 0 0 0-3 3 3 3 0 0 0 3 3 3 3 0 0 0 3-3V6a3 3 0 0 0-3-3 3 3 0 0 0-3 3 3 3 0 0 0 3 3h12a3 3 0 0 0 3-3 3 3 0 0 0-3-3z"
        })]); };
      }

    };

    var CompassIcon = {
      name: 'CompassIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-compass"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("polygon", {
          "points": "16.24 7.76 14.12 14.12 7.76 16.24 9.88 9.88 16.24 7.76"
        })]); };
      }

    };

    var CopyIcon = {
      name: 'CopyIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-copy"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "9",
          "y": "9",
          "width": "13",
          "height": "13",
          "rx": "2",
          "ry": "2"
        }), createVNode("path", {
          "d": "M5 15H4a2 2 0 0 1-2-2V4a2 2 0 0 1 2-2h9a2 2 0 0 1 2 2v1"
        })]); };
      }

    };

    var CornerDownLeftIcon = {
      name: 'CornerDownLeftIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-corner-down-left"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "9 10 4 15 9 20"
        }), createVNode("path", {
          "d": "M20 4v7a4 4 0 0 1-4 4H4"
        })]); };
      }

    };

    var CornerDownRightIcon = {
      name: 'CornerDownRightIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-corner-down-right"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "15 10 20 15 15 20"
        }), createVNode("path", {
          "d": "M4 4v7a4 4 0 0 0 4 4h12"
        })]); };
      }

    };

    var CornerLeftDownIcon = {
      name: 'CornerLeftDownIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-corner-left-down"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "14 15 9 20 4 15"
        }), createVNode("path", {
          "d": "M20 4h-7a4 4 0 0 0-4 4v12"
        })]); };
      }

    };

    var CornerLeftUpIcon = {
      name: 'CornerLeftUpIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-corner-left-up"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "14 9 9 4 4 9"
        }), createVNode("path", {
          "d": "M20 20h-7a4 4 0 0 1-4-4V4"
        })]); };
      }

    };

    var CornerRightDownIcon = {
      name: 'CornerRightDownIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-corner-right-down"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "10 15 15 20 20 15"
        }), createVNode("path", {
          "d": "M4 4h7a4 4 0 0 1 4 4v12"
        })]); };
      }

    };

    var CornerRightUpIcon = {
      name: 'CornerRightUpIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-corner-right-up"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "10 9 15 4 20 9"
        }), createVNode("path", {
          "d": "M4 20h7a4 4 0 0 0 4-4V4"
        })]); };
      }

    };

    var CornerUpLeftIcon = {
      name: 'CornerUpLeftIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-corner-up-left"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "9 14 4 9 9 4"
        }), createVNode("path", {
          "d": "M20 20v-7a4 4 0 0 0-4-4H4"
        })]); };
      }

    };

    var CornerUpRightIcon = {
      name: 'CornerUpRightIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-corner-up-right"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "15 14 20 9 15 4"
        }), createVNode("path", {
          "d": "M4 20v-7a4 4 0 0 1 4-4h12"
        })]); };
      }

    };

    var CpuIcon = {
      name: 'CpuIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-cpu"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "4",
          "y": "4",
          "width": "16",
          "height": "16",
          "rx": "2",
          "ry": "2"
        }), createVNode("rect", {
          "x": "9",
          "y": "9",
          "width": "6",
          "height": "6"
        }), createVNode("line", {
          "x1": "9",
          "y1": "1",
          "x2": "9",
          "y2": "4"
        }), createVNode("line", {
          "x1": "15",
          "y1": "1",
          "x2": "15",
          "y2": "4"
        }), createVNode("line", {
          "x1": "9",
          "y1": "20",
          "x2": "9",
          "y2": "23"
        }), createVNode("line", {
          "x1": "15",
          "y1": "20",
          "x2": "15",
          "y2": "23"
        }), createVNode("line", {
          "x1": "20",
          "y1": "9",
          "x2": "23",
          "y2": "9"
        }), createVNode("line", {
          "x1": "20",
          "y1": "14",
          "x2": "23",
          "y2": "14"
        }), createVNode("line", {
          "x1": "1",
          "y1": "9",
          "x2": "4",
          "y2": "9"
        }), createVNode("line", {
          "x1": "1",
          "y1": "14",
          "x2": "4",
          "y2": "14"
        })]); };
      }

    };

    var CreditCardIcon = {
      name: 'CreditCardIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-credit-card"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "1",
          "y": "4",
          "width": "22",
          "height": "16",
          "rx": "2",
          "ry": "2"
        }), createVNode("line", {
          "x1": "1",
          "y1": "10",
          "x2": "23",
          "y2": "10"
        })]); };
      }

    };

    var CropIcon = {
      name: 'CropIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-crop"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M6.13 1L6 16a2 2 0 0 0 2 2h15"
        }), createVNode("path", {
          "d": "M1 6.13L16 6a2 2 0 0 1 2 2v15"
        })]); };
      }

    };

    var CrosshairIcon = {
      name: 'CrosshairIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-crosshair"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("line", {
          "x1": "22",
          "y1": "12",
          "x2": "18",
          "y2": "12"
        }), createVNode("line", {
          "x1": "6",
          "y1": "12",
          "x2": "2",
          "y2": "12"
        }), createVNode("line", {
          "x1": "12",
          "y1": "6",
          "x2": "12",
          "y2": "2"
        }), createVNode("line", {
          "x1": "12",
          "y1": "22",
          "x2": "12",
          "y2": "18"
        })]); };
      }

    };

    var DatabaseIcon = {
      name: 'DatabaseIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-database"
        }, Object.assign({}, attrs)), [createVNode("ellipse", {
          "cx": "12",
          "cy": "5",
          "rx": "9",
          "ry": "3"
        }), createVNode("path", {
          "d": "M21 12c0 1.66-4 3-9 3s-9-1.34-9-3"
        }), createVNode("path", {
          "d": "M3 5v14c0 1.66 4 3 9 3s9-1.34 9-3V5"
        })]); };
      }

    };

    var DeleteIcon = {
      name: 'DeleteIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-delete"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M21 4H8l-7 8 7 8h13a2 2 0 0 0 2-2V6a2 2 0 0 0-2-2z"
        }), createVNode("line", {
          "x1": "18",
          "y1": "9",
          "x2": "12",
          "y2": "15"
        }), createVNode("line", {
          "x1": "12",
          "y1": "9",
          "x2": "18",
          "y2": "15"
        })]); };
      }

    };

    var DiscIcon = {
      name: 'DiscIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-disc"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "3"
        })]); };
      }

    };

    var DivideCircleIcon = {
      name: 'DivideCircleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-divide-circle"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "8",
          "y1": "12",
          "x2": "16",
          "y2": "12"
        }), createVNode("line", {
          "x1": "12",
          "y1": "16",
          "x2": "12",
          "y2": "16"
        }), createVNode("line", {
          "x1": "12",
          "y1": "8",
          "x2": "12",
          "y2": "8"
        }), createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        })]); };
      }

    };

    var DivideSquareIcon = {
      name: 'DivideSquareIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-divide-square"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "3",
          "y": "3",
          "width": "18",
          "height": "18",
          "rx": "2",
          "ry": "2"
        }), createVNode("line", {
          "x1": "8",
          "y1": "12",
          "x2": "16",
          "y2": "12"
        }), createVNode("line", {
          "x1": "12",
          "y1": "16",
          "x2": "12",
          "y2": "16"
        }), createVNode("line", {
          "x1": "12",
          "y1": "8",
          "x2": "12",
          "y2": "8"
        })]); };
      }

    };

    var DivideIcon = {
      name: 'DivideIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-divide"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "6",
          "r": "2"
        }), createVNode("line", {
          "x1": "5",
          "y1": "12",
          "x2": "19",
          "y2": "12"
        }), createVNode("circle", {
          "cx": "12",
          "cy": "18",
          "r": "2"
        })]); };
      }

    };

    var DollarSignIcon = {
      name: 'DollarSignIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-dollar-sign"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "12",
          "y1": "1",
          "x2": "12",
          "y2": "23"
        }), createVNode("path", {
          "d": "M17 5H9.5a3.5 3.5 0 0 0 0 7h5a3.5 3.5 0 0 1 0 7H6"
        })]); };
      }

    };

    var DownloadCloudIcon = {
      name: 'DownloadCloudIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-download-cloud"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "8 17 12 21 16 17"
        }), createVNode("line", {
          "x1": "12",
          "y1": "12",
          "x2": "12",
          "y2": "21"
        }), createVNode("path", {
          "d": "M20.88 18.09A5 5 0 0 0 18 9h-1.26A8 8 0 1 0 3 16.29"
        })]); };
      }

    };

    var DownloadIcon = {
      name: 'DownloadIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-download"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"
        }), createVNode("polyline", {
          "points": "7 10 12 15 17 10"
        }), createVNode("line", {
          "x1": "12",
          "y1": "15",
          "x2": "12",
          "y2": "3"
        })]); };
      }

    };

    var DribbbleIcon = {
      name: 'DribbbleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-dribbble"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("path", {
          "d": "M8.56 2.75c4.37 6.03 6.02 9.42 8.03 17.72m2.54-15.38c-3.72 4.35-8.94 5.66-16.88 5.85m19.5 1.9c-3.5-.93-6.63-.82-8.94 0-2.58.92-5.01 2.86-7.44 6.32"
        })]); };
      }

    };

    var DropletIcon = {
      name: 'DropletIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-droplet"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M12 2.69l5.66 5.66a8 8 0 1 1-11.31 0z"
        })]); };
      }

    };

    var Edit2Icon = {
      name: 'Edit2Icon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-edit-2"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M17 3a2.828 2.828 0 1 1 4 4L7.5 20.5 2 22l1.5-5.5L17 3z"
        })]); };
      }

    };

    var Edit3Icon = {
      name: 'Edit3Icon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-edit-3"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M12 20h9"
        }), createVNode("path", {
          "d": "M16.5 3.5a2.121 2.121 0 0 1 3 3L7 19l-4 1 1-4L16.5 3.5z"
        })]); };
      }

    };

    var EditIcon = {
      name: 'EditIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-edit"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"
        }), createVNode("path", {
          "d": "M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"
        })]); };
      }

    };

    var ExternalLinkIcon = {
      name: 'ExternalLinkIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-external-link"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M18 13v6a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h6"
        }), createVNode("polyline", {
          "points": "15 3 21 3 21 9"
        }), createVNode("line", {
          "x1": "10",
          "y1": "14",
          "x2": "21",
          "y2": "3"
        })]); };
      }

    };

    var EyeOffIcon = {
      name: 'EyeOffIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-eye-off"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M17.94 17.94A10.07 10.07 0 0 1 12 20c-7 0-11-8-11-8a18.45 18.45 0 0 1 5.06-5.94M9.9 4.24A9.12 9.12 0 0 1 12 4c7 0 11 8 11 8a18.5 18.5 0 0 1-2.16 3.19m-6.72-1.07a3 3 0 1 1-4.24-4.24"
        }), createVNode("line", {
          "x1": "1",
          "y1": "1",
          "x2": "23",
          "y2": "23"
        })]); };
      }

    };

    var EyeIcon = {
      name: 'EyeIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-eye"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"
        }), createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "3"
        })]); };
      }

    };

    var FacebookIcon = {
      name: 'FacebookIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-facebook"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M18 2h-3a5 5 0 0 0-5 5v3H7v4h3v8h4v-8h3l1-4h-4V7a1 1 0 0 1 1-1h3z"
        })]); };
      }

    };

    var FastForwardIcon = {
      name: 'FastForwardIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-fast-forward"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "13 19 22 12 13 5 13 19"
        }), createVNode("polygon", {
          "points": "2 19 11 12 2 5 2 19"
        })]); };
      }

    };

    var FeatherIcon = {
      name: 'FeatherIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-feather"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M20.24 12.24a6 6 0 0 0-8.49-8.49L5 10.5V19h8.5z"
        }), createVNode("line", {
          "x1": "16",
          "y1": "8",
          "x2": "2",
          "y2": "22"
        }), createVNode("line", {
          "x1": "17.5",
          "y1": "15",
          "x2": "9",
          "y2": "15"
        })]); };
      }

    };

    var FigmaIcon = {
      name: 'FigmaIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-figma"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M5 5.5A3.5 3.5 0 0 1 8.5 2H12v7H8.5A3.5 3.5 0 0 1 5 5.5z"
        }), createVNode("path", {
          "d": "M12 2h3.5a3.5 3.5 0 1 1 0 7H12V2z"
        }), createVNode("path", {
          "d": "M12 12.5a3.5 3.5 0 1 1 7 0 3.5 3.5 0 1 1-7 0z"
        }), createVNode("path", {
          "d": "M5 19.5A3.5 3.5 0 0 1 8.5 16H12v3.5a3.5 3.5 0 1 1-7 0z"
        }), createVNode("path", {
          "d": "M5 12.5A3.5 3.5 0 0 1 8.5 9H12v7H8.5A3.5 3.5 0 0 1 5 12.5z"
        })]); };
      }

    };

    var FileMinusIcon = {
      name: 'FileMinusIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-file-minus"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"
        }), createVNode("polyline", {
          "points": "14 2 14 8 20 8"
        }), createVNode("line", {
          "x1": "9",
          "y1": "15",
          "x2": "15",
          "y2": "15"
        })]); };
      }

    };

    var FilePlusIcon = {
      name: 'FilePlusIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-file-plus"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"
        }), createVNode("polyline", {
          "points": "14 2 14 8 20 8"
        }), createVNode("line", {
          "x1": "12",
          "y1": "18",
          "x2": "12",
          "y2": "12"
        }), createVNode("line", {
          "x1": "9",
          "y1": "15",
          "x2": "15",
          "y2": "15"
        })]); };
      }

    };

    var FileTextIcon = {
      name: 'FileTextIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-file-text"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"
        }), createVNode("polyline", {
          "points": "14 2 14 8 20 8"
        }), createVNode("line", {
          "x1": "16",
          "y1": "13",
          "x2": "8",
          "y2": "13"
        }), createVNode("line", {
          "x1": "16",
          "y1": "17",
          "x2": "8",
          "y2": "17"
        }), createVNode("polyline", {
          "points": "10 9 9 9 8 9"
        })]); };
      }

    };

    var FileIcon = {
      name: 'FileIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-file"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M13 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V9z"
        }), createVNode("polyline", {
          "points": "13 2 13 9 20 9"
        })]); };
      }

    };

    var FilmIcon = {
      name: 'FilmIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-film"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "2",
          "y": "2",
          "width": "20",
          "height": "20",
          "rx": "2.18",
          "ry": "2.18"
        }), createVNode("line", {
          "x1": "7",
          "y1": "2",
          "x2": "7",
          "y2": "22"
        }), createVNode("line", {
          "x1": "17",
          "y1": "2",
          "x2": "17",
          "y2": "22"
        }), createVNode("line", {
          "x1": "2",
          "y1": "12",
          "x2": "22",
          "y2": "12"
        }), createVNode("line", {
          "x1": "2",
          "y1": "7",
          "x2": "7",
          "y2": "7"
        }), createVNode("line", {
          "x1": "2",
          "y1": "17",
          "x2": "7",
          "y2": "17"
        }), createVNode("line", {
          "x1": "17",
          "y1": "17",
          "x2": "22",
          "y2": "17"
        }), createVNode("line", {
          "x1": "17",
          "y1": "7",
          "x2": "22",
          "y2": "7"
        })]); };
      }

    };

    var FilterIcon = {
      name: 'FilterIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-filter"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "22 3 2 3 10 12.46 10 19 14 21 14 12.46 22 3"
        })]); };
      }

    };

    var FlagIcon = {
      name: 'FlagIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-flag"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M4 15s1-1 4-1 5 2 8 2 4-1 4-1V3s-1 1-4 1-5-2-8-2-4 1-4 1z"
        }), createVNode("line", {
          "x1": "4",
          "y1": "22",
          "x2": "4",
          "y2": "15"
        })]); };
      }

    };

    var FolderMinusIcon = {
      name: 'FolderMinusIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-folder-minus"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M22 19a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h5l2 3h9a2 2 0 0 1 2 2z"
        }), createVNode("line", {
          "x1": "9",
          "y1": "14",
          "x2": "15",
          "y2": "14"
        })]); };
      }

    };

    var FolderPlusIcon = {
      name: 'FolderPlusIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-folder-plus"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M22 19a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h5l2 3h9a2 2 0 0 1 2 2z"
        }), createVNode("line", {
          "x1": "12",
          "y1": "11",
          "x2": "12",
          "y2": "17"
        }), createVNode("line", {
          "x1": "9",
          "y1": "14",
          "x2": "15",
          "y2": "14"
        })]); };
      }

    };

    var FolderIcon = {
      name: 'FolderIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-folder"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M22 19a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h5l2 3h9a2 2 0 0 1 2 2z"
        })]); };
      }

    };

    var FramerIcon = {
      name: 'FramerIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-framer"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M5 16V9h14V2H5l14 14h-7m-7 0l7 7v-7m-7 0h7"
        })]); };
      }

    };

    var FrownIcon = {
      name: 'FrownIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-frown"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("path", {
          "d": "M16 16s-1.5-2-4-2-4 2-4 2"
        }), createVNode("line", {
          "x1": "9",
          "y1": "9",
          "x2": "9.01",
          "y2": "9"
        }), createVNode("line", {
          "x1": "15",
          "y1": "9",
          "x2": "15.01",
          "y2": "9"
        })]); };
      }

    };

    var GiftIcon = {
      name: 'GiftIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-gift"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "20 12 20 22 4 22 4 12"
        }), createVNode("rect", {
          "x": "2",
          "y": "7",
          "width": "20",
          "height": "5"
        }), createVNode("line", {
          "x1": "12",
          "y1": "22",
          "x2": "12",
          "y2": "7"
        }), createVNode("path", {
          "d": "M12 7H7.5a2.5 2.5 0 0 1 0-5C11 2 12 7 12 7z"
        }), createVNode("path", {
          "d": "M12 7h4.5a2.5 2.5 0 0 0 0-5C13 2 12 7 12 7z"
        })]); };
      }

    };

    var GitBranchIcon = {
      name: 'GitBranchIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-git-branch"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "6",
          "y1": "3",
          "x2": "6",
          "y2": "15"
        }), createVNode("circle", {
          "cx": "18",
          "cy": "6",
          "r": "3"
        }), createVNode("circle", {
          "cx": "6",
          "cy": "18",
          "r": "3"
        }), createVNode("path", {
          "d": "M18 9a9 9 0 0 1-9 9"
        })]); };
      }

    };

    var GitCommitIcon = {
      name: 'GitCommitIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-git-commit"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "4"
        }), createVNode("line", {
          "x1": "1.05",
          "y1": "12",
          "x2": "7",
          "y2": "12"
        }), createVNode("line", {
          "x1": "17.01",
          "y1": "12",
          "x2": "22.96",
          "y2": "12"
        })]); };
      }

    };

    var GitMergeIcon = {
      name: 'GitMergeIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-git-merge"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "18",
          "cy": "18",
          "r": "3"
        }), createVNode("circle", {
          "cx": "6",
          "cy": "6",
          "r": "3"
        }), createVNode("path", {
          "d": "M6 21V9a9 9 0 0 0 9 9"
        })]); };
      }

    };

    var GitPullRequestIcon = {
      name: 'GitPullRequestIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-git-pull-request"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "18",
          "cy": "18",
          "r": "3"
        }), createVNode("circle", {
          "cx": "6",
          "cy": "6",
          "r": "3"
        }), createVNode("path", {
          "d": "M13 6h3a2 2 0 0 1 2 2v7"
        }), createVNode("line", {
          "x1": "6",
          "y1": "9",
          "x2": "6",
          "y2": "21"
        })]); };
      }

    };

    var GithubIcon = {
      name: 'GithubIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-github"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M9 19c-5 1.5-5-2.5-7-3m14 6v-3.87a3.37 3.37 0 0 0-.94-2.61c3.14-.35 6.44-1.54 6.44-7A5.44 5.44 0 0 0 20 4.77 5.07 5.07 0 0 0 19.91 1S18.73.65 16 2.48a13.38 13.38 0 0 0-7 0C6.27.65 5.09 1 5.09 1A5.07 5.07 0 0 0 5 4.77a5.44 5.44 0 0 0-1.5 3.78c0 5.42 3.3 6.61 6.44 7A3.37 3.37 0 0 0 9 18.13V22"
        })]); };
      }

    };

    var GitlabIcon = {
      name: 'GitlabIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-gitlab"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M22.65 14.39L12 22.13 1.35 14.39a.84.84 0 0 1-.3-.94l1.22-3.78 2.44-7.51A.42.42 0 0 1 4.82 2a.43.43 0 0 1 .58 0 .42.42 0 0 1 .11.18l2.44 7.49h8.1l2.44-7.51A.42.42 0 0 1 18.6 2a.43.43 0 0 1 .58 0 .42.42 0 0 1 .11.18l2.44 7.51L23 13.45a.84.84 0 0 1-.35.94z"
        })]); };
      }

    };

    var GlobeIcon = {
      name: 'GlobeIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-globe"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("line", {
          "x1": "2",
          "y1": "12",
          "x2": "22",
          "y2": "12"
        }), createVNode("path", {
          "d": "M12 2a15.3 15.3 0 0 1 4 10 15.3 15.3 0 0 1-4 10 15.3 15.3 0 0 1-4-10 15.3 15.3 0 0 1 4-10z"
        })]); };
      }

    };

    var GridIcon = {
      name: 'GridIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-grid"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "3",
          "y": "3",
          "width": "7",
          "height": "7"
        }), createVNode("rect", {
          "x": "14",
          "y": "3",
          "width": "7",
          "height": "7"
        }), createVNode("rect", {
          "x": "14",
          "y": "14",
          "width": "7",
          "height": "7"
        }), createVNode("rect", {
          "x": "3",
          "y": "14",
          "width": "7",
          "height": "7"
        })]); };
      }

    };

    var HardDriveIcon = {
      name: 'HardDriveIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-hard-drive"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "22",
          "y1": "12",
          "x2": "2",
          "y2": "12"
        }), createVNode("path", {
          "d": "M5.45 5.11L2 12v6a2 2 0 0 0 2 2h16a2 2 0 0 0 2-2v-6l-3.45-6.89A2 2 0 0 0 16.76 4H7.24a2 2 0 0 0-1.79 1.11z"
        }), createVNode("line", {
          "x1": "6",
          "y1": "16",
          "x2": "6.01",
          "y2": "16"
        }), createVNode("line", {
          "x1": "10",
          "y1": "16",
          "x2": "10.01",
          "y2": "16"
        })]); };
      }

    };

    var HashIcon = {
      name: 'HashIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-hash"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "4",
          "y1": "9",
          "x2": "20",
          "y2": "9"
        }), createVNode("line", {
          "x1": "4",
          "y1": "15",
          "x2": "20",
          "y2": "15"
        }), createVNode("line", {
          "x1": "10",
          "y1": "3",
          "x2": "8",
          "y2": "21"
        }), createVNode("line", {
          "x1": "16",
          "y1": "3",
          "x2": "14",
          "y2": "21"
        })]); };
      }

    };

    var HeadphonesIcon = {
      name: 'HeadphonesIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-headphones"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M3 18v-6a9 9 0 0 1 18 0v6"
        }), createVNode("path", {
          "d": "M21 19a2 2 0 0 1-2 2h-1a2 2 0 0 1-2-2v-3a2 2 0 0 1 2-2h3zM3 19a2 2 0 0 0 2 2h1a2 2 0 0 0 2-2v-3a2 2 0 0 0-2-2H3z"
        })]); };
      }

    };

    var HeartIcon = {
      name: 'HeartIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-heart"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M20.84 4.61a5.5 5.5 0 0 0-7.78 0L12 5.67l-1.06-1.06a5.5 5.5 0 0 0-7.78 7.78l1.06 1.06L12 21.23l7.78-7.78 1.06-1.06a5.5 5.5 0 0 0 0-7.78z"
        })]); };
      }

    };

    var HelpCircleIcon = {
      name: 'HelpCircleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-help-circle"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("path", {
          "d": "M9.09 9a3 3 0 0 1 5.83 1c0 2-3 3-3 3"
        }), createVNode("line", {
          "x1": "12",
          "y1": "17",
          "x2": "12.01",
          "y2": "17"
        })]); };
      }

    };

    var HexagonIcon = {
      name: 'HexagonIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-hexagon"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"
        })]); };
      }

    };

    var HomeIcon = {
      name: 'HomeIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-home"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"
        }), createVNode("polyline", {
          "points": "9 22 9 12 15 12 15 22"
        })]); };
      }

    };

    var ImageIcon = {
      name: 'ImageIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-image"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "3",
          "y": "3",
          "width": "18",
          "height": "18",
          "rx": "2",
          "ry": "2"
        }), createVNode("circle", {
          "cx": "8.5",
          "cy": "8.5",
          "r": "1.5"
        }), createVNode("polyline", {
          "points": "21 15 16 10 5 21"
        })]); };
      }

    };

    var InboxIcon = {
      name: 'InboxIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-inbox"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "22 12 16 12 14 15 10 15 8 12 2 12"
        }), createVNode("path", {
          "d": "M5.45 5.11L2 12v6a2 2 0 0 0 2 2h16a2 2 0 0 0 2-2v-6l-3.45-6.89A2 2 0 0 0 16.76 4H7.24a2 2 0 0 0-1.79 1.11z"
        })]); };
      }

    };

    var InfoIcon = {
      name: 'InfoIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-info"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("line", {
          "x1": "12",
          "y1": "16",
          "x2": "12",
          "y2": "12"
        }), createVNode("line", {
          "x1": "12",
          "y1": "8",
          "x2": "12.01",
          "y2": "8"
        })]); };
      }

    };

    var InstagramIcon = {
      name: 'InstagramIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-instagram"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "2",
          "y": "2",
          "width": "20",
          "height": "20",
          "rx": "5",
          "ry": "5"
        }), createVNode("path", {
          "d": "M16 11.37A4 4 0 1 1 12.63 8 4 4 0 0 1 16 11.37z"
        }), createVNode("line", {
          "x1": "17.5",
          "y1": "6.5",
          "x2": "17.51",
          "y2": "6.5"
        })]); };
      }

    };

    var ItalicIcon = {
      name: 'ItalicIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-italic"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "19",
          "y1": "4",
          "x2": "10",
          "y2": "4"
        }), createVNode("line", {
          "x1": "14",
          "y1": "20",
          "x2": "5",
          "y2": "20"
        }), createVNode("line", {
          "x1": "15",
          "y1": "4",
          "x2": "9",
          "y2": "20"
        })]); };
      }

    };

    var KeyIcon = {
      name: 'KeyIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-key"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M21 2l-2 2m-7.61 7.61a5.5 5.5 0 1 1-7.778 7.778 5.5 5.5 0 0 1 7.777-7.777zm0 0L15.5 7.5m0 0l3 3L22 7l-3-3m-3.5 3.5L19 4"
        })]); };
      }

    };

    var LayersIcon = {
      name: 'LayersIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-layers"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "12 2 2 7 12 12 22 7 12 2"
        }), createVNode("polyline", {
          "points": "2 17 12 22 22 17"
        }), createVNode("polyline", {
          "points": "2 12 12 17 22 12"
        })]); };
      }

    };

    var LayoutIcon = {
      name: 'LayoutIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-layout"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "3",
          "y": "3",
          "width": "18",
          "height": "18",
          "rx": "2",
          "ry": "2"
        }), createVNode("line", {
          "x1": "3",
          "y1": "9",
          "x2": "21",
          "y2": "9"
        }), createVNode("line", {
          "x1": "9",
          "y1": "21",
          "x2": "9",
          "y2": "9"
        })]); };
      }

    };

    var LifeBuoyIcon = {
      name: 'LifeBuoyIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-life-buoy"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "4"
        }), createVNode("line", {
          "x1": "4.93",
          "y1": "4.93",
          "x2": "9.17",
          "y2": "9.17"
        }), createVNode("line", {
          "x1": "14.83",
          "y1": "14.83",
          "x2": "19.07",
          "y2": "19.07"
        }), createVNode("line", {
          "x1": "14.83",
          "y1": "9.17",
          "x2": "19.07",
          "y2": "4.93"
        }), createVNode("line", {
          "x1": "14.83",
          "y1": "9.17",
          "x2": "18.36",
          "y2": "5.64"
        }), createVNode("line", {
          "x1": "4.93",
          "y1": "19.07",
          "x2": "9.17",
          "y2": "14.83"
        })]); };
      }

    };

    var Link2Icon = {
      name: 'Link2Icon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-link-2"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M15 7h3a5 5 0 0 1 5 5 5 5 0 0 1-5 5h-3m-6 0H6a5 5 0 0 1-5-5 5 5 0 0 1 5-5h3"
        }), createVNode("line", {
          "x1": "8",
          "y1": "12",
          "x2": "16",
          "y2": "12"
        })]); };
      }

    };

    var LinkIcon = {
      name: 'LinkIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-link"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M10 13a5 5 0 0 0 7.54.54l3-3a5 5 0 0 0-7.07-7.07l-1.72 1.71"
        }), createVNode("path", {
          "d": "M14 11a5 5 0 0 0-7.54-.54l-3 3a5 5 0 0 0 7.07 7.07l1.71-1.71"
        })]); };
      }

    };

    var LinkedinIcon = {
      name: 'LinkedinIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-linkedin"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M16 8a6 6 0 0 1 6 6v7h-4v-7a2 2 0 0 0-2-2 2 2 0 0 0-2 2v7h-4v-7a6 6 0 0 1 6-6z"
        }), createVNode("rect", {
          "x": "2",
          "y": "9",
          "width": "4",
          "height": "12"
        }), createVNode("circle", {
          "cx": "4",
          "cy": "4",
          "r": "2"
        })]); };
      }

    };

    var ListIcon = {
      name: 'ListIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-list"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "8",
          "y1": "6",
          "x2": "21",
          "y2": "6"
        }), createVNode("line", {
          "x1": "8",
          "y1": "12",
          "x2": "21",
          "y2": "12"
        }), createVNode("line", {
          "x1": "8",
          "y1": "18",
          "x2": "21",
          "y2": "18"
        }), createVNode("line", {
          "x1": "3",
          "y1": "6",
          "x2": "3.01",
          "y2": "6"
        }), createVNode("line", {
          "x1": "3",
          "y1": "12",
          "x2": "3.01",
          "y2": "12"
        }), createVNode("line", {
          "x1": "3",
          "y1": "18",
          "x2": "3.01",
          "y2": "18"
        })]); };
      }

    };

    var LoaderIcon = {
      name: 'LoaderIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-loader"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "12",
          "y1": "2",
          "x2": "12",
          "y2": "6"
        }), createVNode("line", {
          "x1": "12",
          "y1": "18",
          "x2": "12",
          "y2": "22"
        }), createVNode("line", {
          "x1": "4.93",
          "y1": "4.93",
          "x2": "7.76",
          "y2": "7.76"
        }), createVNode("line", {
          "x1": "16.24",
          "y1": "16.24",
          "x2": "19.07",
          "y2": "19.07"
        }), createVNode("line", {
          "x1": "2",
          "y1": "12",
          "x2": "6",
          "y2": "12"
        }), createVNode("line", {
          "x1": "18",
          "y1": "12",
          "x2": "22",
          "y2": "12"
        }), createVNode("line", {
          "x1": "4.93",
          "y1": "19.07",
          "x2": "7.76",
          "y2": "16.24"
        }), createVNode("line", {
          "x1": "16.24",
          "y1": "7.76",
          "x2": "19.07",
          "y2": "4.93"
        })]); };
      }

    };

    var LockIcon = {
      name: 'LockIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-lock"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "3",
          "y": "11",
          "width": "18",
          "height": "11",
          "rx": "2",
          "ry": "2"
        }), createVNode("path", {
          "d": "M7 11V7a5 5 0 0 1 10 0v4"
        })]); };
      }

    };

    var LogInIcon = {
      name: 'LogInIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-log-in"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M15 3h4a2 2 0 0 1 2 2v14a2 2 0 0 1-2 2h-4"
        }), createVNode("polyline", {
          "points": "10 17 15 12 10 7"
        }), createVNode("line", {
          "x1": "15",
          "y1": "12",
          "x2": "3",
          "y2": "12"
        })]); };
      }

    };

    var LogOutIcon = {
      name: 'LogOutIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-log-out"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M9 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h4"
        }), createVNode("polyline", {
          "points": "16 17 21 12 16 7"
        }), createVNode("line", {
          "x1": "21",
          "y1": "12",
          "x2": "9",
          "y2": "12"
        })]); };
      }

    };

    var MailIcon = {
      name: 'MailIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-mail"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M4 4h16c1.1 0 2 .9 2 2v12c0 1.1-.9 2-2 2H4c-1.1 0-2-.9-2-2V6c0-1.1.9-2 2-2z"
        }), createVNode("polyline", {
          "points": "22,6 12,13 2,6"
        })]); };
      }

    };

    var MapPinIcon = {
      name: 'MapPinIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-map-pin"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0 1 18 0z"
        }), createVNode("circle", {
          "cx": "12",
          "cy": "10",
          "r": "3"
        })]); };
      }

    };

    var MapIcon = {
      name: 'MapIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-map"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "1 6 1 22 8 18 16 22 23 18 23 2 16 6 8 2 1 6"
        }), createVNode("line", {
          "x1": "8",
          "y1": "2",
          "x2": "8",
          "y2": "18"
        }), createVNode("line", {
          "x1": "16",
          "y1": "6",
          "x2": "16",
          "y2": "22"
        })]); };
      }

    };

    var Maximize2Icon = {
      name: 'Maximize2Icon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-maximize-2"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "15 3 21 3 21 9"
        }), createVNode("polyline", {
          "points": "9 21 3 21 3 15"
        }), createVNode("line", {
          "x1": "21",
          "y1": "3",
          "x2": "14",
          "y2": "10"
        }), createVNode("line", {
          "x1": "3",
          "y1": "21",
          "x2": "10",
          "y2": "14"
        })]); };
      }

    };

    var MaximizeIcon = {
      name: 'MaximizeIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-maximize"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M8 3H5a2 2 0 0 0-2 2v3m18 0V5a2 2 0 0 0-2-2h-3m0 18h3a2 2 0 0 0 2-2v-3M3 16v3a2 2 0 0 0 2 2h3"
        })]); };
      }

    };

    var MehIcon = {
      name: 'MehIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-meh"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("line", {
          "x1": "8",
          "y1": "15",
          "x2": "16",
          "y2": "15"
        }), createVNode("line", {
          "x1": "9",
          "y1": "9",
          "x2": "9.01",
          "y2": "9"
        }), createVNode("line", {
          "x1": "15",
          "y1": "9",
          "x2": "15.01",
          "y2": "9"
        })]); };
      }

    };

    var MenuIcon = {
      name: 'MenuIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-menu"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "3",
          "y1": "12",
          "x2": "21",
          "y2": "12"
        }), createVNode("line", {
          "x1": "3",
          "y1": "6",
          "x2": "21",
          "y2": "6"
        }), createVNode("line", {
          "x1": "3",
          "y1": "18",
          "x2": "21",
          "y2": "18"
        })]); };
      }

    };

    var MessageCircleIcon = {
      name: 'MessageCircleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-message-circle"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M21 11.5a8.38 8.38 0 0 1-.9 3.8 8.5 8.5 0 0 1-7.6 4.7 8.38 8.38 0 0 1-3.8-.9L3 21l1.9-5.7a8.38 8.38 0 0 1-.9-3.8 8.5 8.5 0 0 1 4.7-7.6 8.38 8.38 0 0 1 3.8-.9h.5a8.48 8.48 0 0 1 8 8v.5z"
        })]); };
      }

    };

    var MessageSquareIcon = {
      name: 'MessageSquareIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-message-square"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"
        })]); };
      }

    };

    var MicOffIcon = {
      name: 'MicOffIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-mic-off"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "1",
          "y1": "1",
          "x2": "23",
          "y2": "23"
        }), createVNode("path", {
          "d": "M9 9v3a3 3 0 0 0 5.12 2.12M15 9.34V4a3 3 0 0 0-5.94-.6"
        }), createVNode("path", {
          "d": "M17 16.95A7 7 0 0 1 5 12v-2m14 0v2a7 7 0 0 1-.11 1.23"
        }), createVNode("line", {
          "x1": "12",
          "y1": "19",
          "x2": "12",
          "y2": "23"
        }), createVNode("line", {
          "x1": "8",
          "y1": "23",
          "x2": "16",
          "y2": "23"
        })]); };
      }

    };

    var MicIcon = {
      name: 'MicIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-mic"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M12 1a3 3 0 0 0-3 3v8a3 3 0 0 0 6 0V4a3 3 0 0 0-3-3z"
        }), createVNode("path", {
          "d": "M19 10v2a7 7 0 0 1-14 0v-2"
        }), createVNode("line", {
          "x1": "12",
          "y1": "19",
          "x2": "12",
          "y2": "23"
        }), createVNode("line", {
          "x1": "8",
          "y1": "23",
          "x2": "16",
          "y2": "23"
        })]); };
      }

    };

    var Minimize2Icon = {
      name: 'Minimize2Icon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-minimize-2"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "4 14 10 14 10 20"
        }), createVNode("polyline", {
          "points": "20 10 14 10 14 4"
        }), createVNode("line", {
          "x1": "14",
          "y1": "10",
          "x2": "21",
          "y2": "3"
        }), createVNode("line", {
          "x1": "3",
          "y1": "21",
          "x2": "10",
          "y2": "14"
        })]); };
      }

    };

    var MinimizeIcon = {
      name: 'MinimizeIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-minimize"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M8 3v3a2 2 0 0 1-2 2H3m18 0h-3a2 2 0 0 1-2-2V3m0 18v-3a2 2 0 0 1 2-2h3M3 16h3a2 2 0 0 1 2 2v3"
        })]); };
      }

    };

    var MinusCircleIcon = {
      name: 'MinusCircleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-minus-circle"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("line", {
          "x1": "8",
          "y1": "12",
          "x2": "16",
          "y2": "12"
        })]); };
      }

    };

    var MinusSquareIcon = {
      name: 'MinusSquareIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-minus-square"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "3",
          "y": "3",
          "width": "18",
          "height": "18",
          "rx": "2",
          "ry": "2"
        }), createVNode("line", {
          "x1": "8",
          "y1": "12",
          "x2": "16",
          "y2": "12"
        })]); };
      }

    };

    var MinusIcon = {
      name: 'MinusIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-minus"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "5",
          "y1": "12",
          "x2": "19",
          "y2": "12"
        })]); };
      }

    };

    var MonitorIcon = {
      name: 'MonitorIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-monitor"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "2",
          "y": "3",
          "width": "20",
          "height": "14",
          "rx": "2",
          "ry": "2"
        }), createVNode("line", {
          "x1": "8",
          "y1": "21",
          "x2": "16",
          "y2": "21"
        }), createVNode("line", {
          "x1": "12",
          "y1": "17",
          "x2": "12",
          "y2": "21"
        })]); };
      }

    };

    var MoonIcon = {
      name: 'MoonIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-moon"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M21 12.79A9 9 0 1 1 11.21 3 7 7 0 0 0 21 12.79z"
        })]); };
      }

    };

    var MoreHorizontalIcon = {
      name: 'MoreHorizontalIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-more-horizontal"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "1"
        }), createVNode("circle", {
          "cx": "19",
          "cy": "12",
          "r": "1"
        }), createVNode("circle", {
          "cx": "5",
          "cy": "12",
          "r": "1"
        })]); };
      }

    };

    var MoreVerticalIcon = {
      name: 'MoreVerticalIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-more-vertical"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "1"
        }), createVNode("circle", {
          "cx": "12",
          "cy": "5",
          "r": "1"
        }), createVNode("circle", {
          "cx": "12",
          "cy": "19",
          "r": "1"
        })]); };
      }

    };

    var MousePointerIcon = {
      name: 'MousePointerIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-mouse-pointer"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M3 3l7.07 16.97 2.51-7.39 7.39-2.51L3 3z"
        }), createVNode("path", {
          "d": "M13 13l6 6"
        })]); };
      }

    };

    var MoveIcon = {
      name: 'MoveIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-move"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "5 9 2 12 5 15"
        }), createVNode("polyline", {
          "points": "9 5 12 2 15 5"
        }), createVNode("polyline", {
          "points": "15 19 12 22 9 19"
        }), createVNode("polyline", {
          "points": "19 9 22 12 19 15"
        }), createVNode("line", {
          "x1": "2",
          "y1": "12",
          "x2": "22",
          "y2": "12"
        }), createVNode("line", {
          "x1": "12",
          "y1": "2",
          "x2": "12",
          "y2": "22"
        })]); };
      }

    };

    var MusicIcon = {
      name: 'MusicIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-music"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M9 18V5l12-2v13"
        }), createVNode("circle", {
          "cx": "6",
          "cy": "18",
          "r": "3"
        }), createVNode("circle", {
          "cx": "18",
          "cy": "16",
          "r": "3"
        })]); };
      }

    };

    var Navigation2Icon = {
      name: 'Navigation2Icon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-navigation-2"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "12 2 19 21 12 17 5 21 12 2"
        })]); };
      }

    };

    var NavigationIcon = {
      name: 'NavigationIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-navigation"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "3 11 22 2 13 21 11 13 3 11"
        })]); };
      }

    };

    var OctagonIcon = {
      name: 'OctagonIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-octagon"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "7.86 2 16.14 2 22 7.86 22 16.14 16.14 22 7.86 22 2 16.14 2 7.86 7.86 2"
        })]); };
      }

    };

    var PackageIcon = {
      name: 'PackageIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-package"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "16.5",
          "y1": "9.4",
          "x2": "7.5",
          "y2": "4.21"
        }), createVNode("path", {
          "d": "M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"
        }), createVNode("polyline", {
          "points": "3.27 6.96 12 12.01 20.73 6.96"
        }), createVNode("line", {
          "x1": "12",
          "y1": "22.08",
          "x2": "12",
          "y2": "12"
        })]); };
      }

    };

    var PaperclipIcon = {
      name: 'PaperclipIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-paperclip"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M21.44 11.05l-9.19 9.19a6 6 0 0 1-8.49-8.49l9.19-9.19a4 4 0 0 1 5.66 5.66l-9.2 9.19a2 2 0 0 1-2.83-2.83l8.49-8.48"
        })]); };
      }

    };

    var PauseCircleIcon = {
      name: 'PauseCircleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-pause-circle"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("line", {
          "x1": "10",
          "y1": "15",
          "x2": "10",
          "y2": "9"
        }), createVNode("line", {
          "x1": "14",
          "y1": "15",
          "x2": "14",
          "y2": "9"
        })]); };
      }

    };

    var PauseIcon = {
      name: 'PauseIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-pause"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "6",
          "y": "4",
          "width": "4",
          "height": "16"
        }), createVNode("rect", {
          "x": "14",
          "y": "4",
          "width": "4",
          "height": "16"
        })]); };
      }

    };

    var PenToolIcon = {
      name: 'PenToolIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-pen-tool"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M12 19l7-7 3 3-7 7-3-3z"
        }), createVNode("path", {
          "d": "M18 13l-1.5-7.5L2 2l3.5 14.5L13 18l5-5z"
        }), createVNode("path", {
          "d": "M2 2l7.586 7.586"
        }), createVNode("circle", {
          "cx": "11",
          "cy": "11",
          "r": "2"
        })]); };
      }

    };

    var PercentIcon = {
      name: 'PercentIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-percent"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "19",
          "y1": "5",
          "x2": "5",
          "y2": "19"
        }), createVNode("circle", {
          "cx": "6.5",
          "cy": "6.5",
          "r": "2.5"
        }), createVNode("circle", {
          "cx": "17.5",
          "cy": "17.5",
          "r": "2.5"
        })]); };
      }

    };

    var PhoneCallIcon = {
      name: 'PhoneCallIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-phone-call"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M15.05 5A5 5 0 0 1 19 8.95M15.05 1A9 9 0 0 1 23 8.94m-1 7.98v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"
        })]); };
      }

    };

    var PhoneForwardedIcon = {
      name: 'PhoneForwardedIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-phone-forwarded"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "19 1 23 5 19 9"
        }), createVNode("line", {
          "x1": "15",
          "y1": "5",
          "x2": "23",
          "y2": "5"
        }), createVNode("path", {
          "d": "M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"
        })]); };
      }

    };

    var PhoneIncomingIcon = {
      name: 'PhoneIncomingIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-phone-incoming"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "16 2 16 8 22 8"
        }), createVNode("line", {
          "x1": "23",
          "y1": "1",
          "x2": "16",
          "y2": "8"
        }), createVNode("path", {
          "d": "M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"
        })]); };
      }

    };

    var PhoneMissedIcon = {
      name: 'PhoneMissedIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-phone-missed"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "23",
          "y1": "1",
          "x2": "17",
          "y2": "7"
        }), createVNode("line", {
          "x1": "17",
          "y1": "1",
          "x2": "23",
          "y2": "7"
        }), createVNode("path", {
          "d": "M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"
        })]); };
      }

    };

    var PhoneOffIcon = {
      name: 'PhoneOffIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-phone-off"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M10.68 13.31a16 16 0 0 0 3.41 2.6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7 2 2 0 0 1 1.72 2v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.42 19.42 0 0 1-3.33-2.67m-2.67-3.34a19.79 19.79 0 0 1-3.07-8.63A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91"
        }), createVNode("line", {
          "x1": "23",
          "y1": "1",
          "x2": "1",
          "y2": "23"
        })]); };
      }

    };

    var PhoneOutgoingIcon = {
      name: 'PhoneOutgoingIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-phone-outgoing"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "23 7 23 1 17 1"
        }), createVNode("line", {
          "x1": "16",
          "y1": "8",
          "x2": "23",
          "y2": "1"
        }), createVNode("path", {
          "d": "M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"
        })]); };
      }

    };

    var PhoneIcon = {
      name: 'PhoneIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-phone"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"
        })]); };
      }

    };

    var PieChartIcon = {
      name: 'PieChartIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-pie-chart"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M21.21 15.89A10 10 0 1 1 8 2.83"
        }), createVNode("path", {
          "d": "M22 12A10 10 0 0 0 12 2v10z"
        })]); };
      }

    };

    var PlayCircleIcon = {
      name: 'PlayCircleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-play-circle"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("polygon", {
          "points": "10 8 16 12 10 16 10 8"
        })]); };
      }

    };

    var PlayIcon = {
      name: 'PlayIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-play"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "5 3 19 12 5 21 5 3"
        })]); };
      }

    };

    var PlusCircleIcon = {
      name: 'PlusCircleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-plus-circle"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("line", {
          "x1": "12",
          "y1": "8",
          "x2": "12",
          "y2": "16"
        }), createVNode("line", {
          "x1": "8",
          "y1": "12",
          "x2": "16",
          "y2": "12"
        })]); };
      }

    };

    var PlusSquareIcon = {
      name: 'PlusSquareIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-plus-square"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "3",
          "y": "3",
          "width": "18",
          "height": "18",
          "rx": "2",
          "ry": "2"
        }), createVNode("line", {
          "x1": "12",
          "y1": "8",
          "x2": "12",
          "y2": "16"
        }), createVNode("line", {
          "x1": "8",
          "y1": "12",
          "x2": "16",
          "y2": "12"
        })]); };
      }

    };

    var PlusIcon = {
      name: 'PlusIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-plus"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "12",
          "y1": "5",
          "x2": "12",
          "y2": "19"
        }), createVNode("line", {
          "x1": "5",
          "y1": "12",
          "x2": "19",
          "y2": "12"
        })]); };
      }

    };

    var PocketIcon = {
      name: 'PocketIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-pocket"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M4 3h16a2 2 0 0 1 2 2v6a10 10 0 0 1-10 10A10 10 0 0 1 2 11V5a2 2 0 0 1 2-2z"
        }), createVNode("polyline", {
          "points": "8 10 12 14 16 10"
        })]); };
      }

    };

    var PowerIcon = {
      name: 'PowerIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-power"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M18.36 6.64a9 9 0 1 1-12.73 0"
        }), createVNode("line", {
          "x1": "12",
          "y1": "2",
          "x2": "12",
          "y2": "12"
        })]); };
      }

    };

    var PrinterIcon = {
      name: 'PrinterIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-printer"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "6 9 6 2 18 2 18 9"
        }), createVNode("path", {
          "d": "M6 18H4a2 2 0 0 1-2-2v-5a2 2 0 0 1 2-2h16a2 2 0 0 1 2 2v5a2 2 0 0 1-2 2h-2"
        }), createVNode("rect", {
          "x": "6",
          "y": "14",
          "width": "12",
          "height": "8"
        })]); };
      }

    };

    var RadioIcon = {
      name: 'RadioIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-radio"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "2"
        }), createVNode("path", {
          "d": "M16.24 7.76a6 6 0 0 1 0 8.49m-8.48-.01a6 6 0 0 1 0-8.49m11.31-2.82a10 10 0 0 1 0 14.14m-14.14 0a10 10 0 0 1 0-14.14"
        })]); };
      }

    };

    var RefreshCcwIcon = {
      name: 'RefreshCcwIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-refresh-ccw"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "1 4 1 10 7 10"
        }), createVNode("polyline", {
          "points": "23 20 23 14 17 14"
        }), createVNode("path", {
          "d": "M20.49 9A9 9 0 0 0 5.64 5.64L1 10m22 4l-4.64 4.36A9 9 0 0 1 3.51 15"
        })]); };
      }

    };

    var RefreshCwIcon = {
      name: 'RefreshCwIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-refresh-cw"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "23 4 23 10 17 10"
        }), createVNode("polyline", {
          "points": "1 20 1 14 7 14"
        }), createVNode("path", {
          "d": "M3.51 9a9 9 0 0 1 14.85-3.36L23 10M1 14l4.64 4.36A9 9 0 0 0 20.49 15"
        })]); };
      }

    };

    var RepeatIcon = {
      name: 'RepeatIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-repeat"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "17 1 21 5 17 9"
        }), createVNode("path", {
          "d": "M3 11V9a4 4 0 0 1 4-4h14"
        }), createVNode("polyline", {
          "points": "7 23 3 19 7 15"
        }), createVNode("path", {
          "d": "M21 13v2a4 4 0 0 1-4 4H3"
        })]); };
      }

    };

    var RewindIcon = {
      name: 'RewindIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-rewind"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "11 19 2 12 11 5 11 19"
        }), createVNode("polygon", {
          "points": "22 19 13 12 22 5 22 19"
        })]); };
      }

    };

    var RotateCcwIcon = {
      name: 'RotateCcwIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-rotate-ccw"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "1 4 1 10 7 10"
        }), createVNode("path", {
          "d": "M3.51 15a9 9 0 1 0 2.13-9.36L1 10"
        })]); };
      }

    };

    var RotateCwIcon = {
      name: 'RotateCwIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-rotate-cw"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "23 4 23 10 17 10"
        }), createVNode("path", {
          "d": "M20.49 15a9 9 0 1 1-2.12-9.36L23 10"
        })]); };
      }

    };

    var RssIcon = {
      name: 'RssIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-rss"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M4 11a9 9 0 0 1 9 9"
        }), createVNode("path", {
          "d": "M4 4a16 16 0 0 1 16 16"
        }), createVNode("circle", {
          "cx": "5",
          "cy": "19",
          "r": "1"
        })]); };
      }

    };

    var SaveIcon = {
      name: 'SaveIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-save"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M19 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11l5 5v11a2 2 0 0 1-2 2z"
        }), createVNode("polyline", {
          "points": "17 21 17 13 7 13 7 21"
        }), createVNode("polyline", {
          "points": "7 3 7 8 15 8"
        })]); };
      }

    };

    var ScissorsIcon = {
      name: 'ScissorsIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-scissors"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "6",
          "cy": "6",
          "r": "3"
        }), createVNode("circle", {
          "cx": "6",
          "cy": "18",
          "r": "3"
        }), createVNode("line", {
          "x1": "20",
          "y1": "4",
          "x2": "8.12",
          "y2": "15.88"
        }), createVNode("line", {
          "x1": "14.47",
          "y1": "14.48",
          "x2": "20",
          "y2": "20"
        }), createVNode("line", {
          "x1": "8.12",
          "y1": "8.12",
          "x2": "12",
          "y2": "12"
        })]); };
      }

    };

    var SearchIcon = {
      name: 'SearchIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-search"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "11",
          "cy": "11",
          "r": "8"
        }), createVNode("line", {
          "x1": "21",
          "y1": "21",
          "x2": "16.65",
          "y2": "16.65"
        })]); };
      }

    };

    var SendIcon = {
      name: 'SendIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-send"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "22",
          "y1": "2",
          "x2": "11",
          "y2": "13"
        }), createVNode("polygon", {
          "points": "22 2 15 22 11 13 2 9 22 2"
        })]); };
      }

    };

    var ServerIcon = {
      name: 'ServerIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-server"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "2",
          "y": "2",
          "width": "20",
          "height": "8",
          "rx": "2",
          "ry": "2"
        }), createVNode("rect", {
          "x": "2",
          "y": "14",
          "width": "20",
          "height": "8",
          "rx": "2",
          "ry": "2"
        }), createVNode("line", {
          "x1": "6",
          "y1": "6",
          "x2": "6.01",
          "y2": "6"
        }), createVNode("line", {
          "x1": "6",
          "y1": "18",
          "x2": "6.01",
          "y2": "18"
        })]); };
      }

    };

    var SettingsIcon = {
      name: 'SettingsIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-settings"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "3"
        }), createVNode("path", {
          "d": "M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0l.06.06a1.65 1.65 0 0 0 1.82.33H9a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83l-.06.06a1.65 1.65 0 0 0-.33 1.82V9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09a1.65 1.65 0 0 0-1.51 1z"
        })]); };
      }

    };

    var Share2Icon = {
      name: 'Share2Icon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-share-2"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "18",
          "cy": "5",
          "r": "3"
        }), createVNode("circle", {
          "cx": "6",
          "cy": "12",
          "r": "3"
        }), createVNode("circle", {
          "cx": "18",
          "cy": "19",
          "r": "3"
        }), createVNode("line", {
          "x1": "8.59",
          "y1": "13.51",
          "x2": "15.42",
          "y2": "17.49"
        }), createVNode("line", {
          "x1": "15.41",
          "y1": "6.51",
          "x2": "8.59",
          "y2": "10.49"
        })]); };
      }

    };

    var ShareIcon = {
      name: 'ShareIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-share"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M4 12v8a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2v-8"
        }), createVNode("polyline", {
          "points": "16 6 12 2 8 6"
        }), createVNode("line", {
          "x1": "12",
          "y1": "2",
          "x2": "12",
          "y2": "15"
        })]); };
      }

    };

    var ShieldOffIcon = {
      name: 'ShieldOffIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-shield-off"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M19.69 14a6.9 6.9 0 0 0 .31-2V5l-8-3-3.16 1.18"
        }), createVNode("path", {
          "d": "M4.73 4.73L4 5v7c0 6 8 10 8 10a20.29 20.29 0 0 0 5.62-4.38"
        }), createVNode("line", {
          "x1": "1",
          "y1": "1",
          "x2": "23",
          "y2": "23"
        })]); };
      }

    };

    var ShieldIcon = {
      name: 'ShieldIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-shield"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"
        })]); };
      }

    };

    var ShoppingBagIcon = {
      name: 'ShoppingBagIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-shopping-bag"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M6 2L3 6v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2V6l-3-4z"
        }), createVNode("line", {
          "x1": "3",
          "y1": "6",
          "x2": "21",
          "y2": "6"
        }), createVNode("path", {
          "d": "M16 10a4 4 0 0 1-8 0"
        })]); };
      }

    };

    var ShoppingCartIcon = {
      name: 'ShoppingCartIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-shopping-cart"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "9",
          "cy": "21",
          "r": "1"
        }), createVNode("circle", {
          "cx": "20",
          "cy": "21",
          "r": "1"
        }), createVNode("path", {
          "d": "M1 1h4l2.68 13.39a2 2 0 0 0 2 1.61h9.72a2 2 0 0 0 2-1.61L23 6H6"
        })]); };
      }

    };

    var ShuffleIcon = {
      name: 'ShuffleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-shuffle"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "16 3 21 3 21 8"
        }), createVNode("line", {
          "x1": "4",
          "y1": "20",
          "x2": "21",
          "y2": "3"
        }), createVNode("polyline", {
          "points": "21 16 21 21 16 21"
        }), createVNode("line", {
          "x1": "15",
          "y1": "15",
          "x2": "21",
          "y2": "21"
        }), createVNode("line", {
          "x1": "4",
          "y1": "4",
          "x2": "9",
          "y2": "9"
        })]); };
      }

    };

    var SidebarIcon = {
      name: 'SidebarIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-sidebar"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "3",
          "y": "3",
          "width": "18",
          "height": "18",
          "rx": "2",
          "ry": "2"
        }), createVNode("line", {
          "x1": "9",
          "y1": "3",
          "x2": "9",
          "y2": "21"
        })]); };
      }

    };

    var SkipBackIcon = {
      name: 'SkipBackIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-skip-back"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "19 20 9 12 19 4 19 20"
        }), createVNode("line", {
          "x1": "5",
          "y1": "19",
          "x2": "5",
          "y2": "5"
        })]); };
      }

    };

    var SkipForwardIcon = {
      name: 'SkipForwardIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-skip-forward"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "5 4 15 12 5 20 5 4"
        }), createVNode("line", {
          "x1": "19",
          "y1": "5",
          "x2": "19",
          "y2": "19"
        })]); };
      }

    };

    var SlackIcon = {
      name: 'SlackIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-slack"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M14.5 10c-.83 0-1.5-.67-1.5-1.5v-5c0-.83.67-1.5 1.5-1.5s1.5.67 1.5 1.5v5c0 .83-.67 1.5-1.5 1.5z"
        }), createVNode("path", {
          "d": "M20.5 10H19V8.5c0-.83.67-1.5 1.5-1.5s1.5.67 1.5 1.5-.67 1.5-1.5 1.5z"
        }), createVNode("path", {
          "d": "M9.5 14c.83 0 1.5.67 1.5 1.5v5c0 .83-.67 1.5-1.5 1.5S8 21.33 8 20.5v-5c0-.83.67-1.5 1.5-1.5z"
        }), createVNode("path", {
          "d": "M3.5 14H5v1.5c0 .83-.67 1.5-1.5 1.5S2 16.33 2 15.5 2.67 14 3.5 14z"
        }), createVNode("path", {
          "d": "M14 14.5c0-.83.67-1.5 1.5-1.5h5c.83 0 1.5.67 1.5 1.5s-.67 1.5-1.5 1.5h-5c-.83 0-1.5-.67-1.5-1.5z"
        }), createVNode("path", {
          "d": "M15.5 19H14v1.5c0 .83.67 1.5 1.5 1.5s1.5-.67 1.5-1.5-.67-1.5-1.5-1.5z"
        }), createVNode("path", {
          "d": "M10 9.5C10 8.67 9.33 8 8.5 8h-5C2.67 8 2 8.67 2 9.5S2.67 11 3.5 11h5c.83 0 1.5-.67 1.5-1.5z"
        }), createVNode("path", {
          "d": "M8.5 5H10V3.5C10 2.67 9.33 2 8.5 2S7 2.67 7 3.5 7.67 5 8.5 5z"
        })]); };
      }

    };

    var SlashIcon = {
      name: 'SlashIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-slash"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("line", {
          "x1": "4.93",
          "y1": "4.93",
          "x2": "19.07",
          "y2": "19.07"
        })]); };
      }

    };

    var SlidersIcon = {
      name: 'SlidersIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-sliders"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "4",
          "y1": "21",
          "x2": "4",
          "y2": "14"
        }), createVNode("line", {
          "x1": "4",
          "y1": "10",
          "x2": "4",
          "y2": "3"
        }), createVNode("line", {
          "x1": "12",
          "y1": "21",
          "x2": "12",
          "y2": "12"
        }), createVNode("line", {
          "x1": "12",
          "y1": "8",
          "x2": "12",
          "y2": "3"
        }), createVNode("line", {
          "x1": "20",
          "y1": "21",
          "x2": "20",
          "y2": "16"
        }), createVNode("line", {
          "x1": "20",
          "y1": "12",
          "x2": "20",
          "y2": "3"
        }), createVNode("line", {
          "x1": "1",
          "y1": "14",
          "x2": "7",
          "y2": "14"
        }), createVNode("line", {
          "x1": "9",
          "y1": "8",
          "x2": "15",
          "y2": "8"
        }), createVNode("line", {
          "x1": "17",
          "y1": "16",
          "x2": "23",
          "y2": "16"
        })]); };
      }

    };

    var SmartphoneIcon = {
      name: 'SmartphoneIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-smartphone"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "5",
          "y": "2",
          "width": "14",
          "height": "20",
          "rx": "2",
          "ry": "2"
        }), createVNode("line", {
          "x1": "12",
          "y1": "18",
          "x2": "12.01",
          "y2": "18"
        })]); };
      }

    };

    var SmileIcon = {
      name: 'SmileIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-smile"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("path", {
          "d": "M8 14s1.5 2 4 2 4-2 4-2"
        }), createVNode("line", {
          "x1": "9",
          "y1": "9",
          "x2": "9.01",
          "y2": "9"
        }), createVNode("line", {
          "x1": "15",
          "y1": "9",
          "x2": "15.01",
          "y2": "9"
        })]); };
      }

    };

    var SpeakerIcon = {
      name: 'SpeakerIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-speaker"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "4",
          "y": "2",
          "width": "16",
          "height": "20",
          "rx": "2",
          "ry": "2"
        }), createVNode("circle", {
          "cx": "12",
          "cy": "14",
          "r": "4"
        }), createVNode("line", {
          "x1": "12",
          "y1": "6",
          "x2": "12.01",
          "y2": "6"
        })]); };
      }

    };

    var SquareIcon = {
      name: 'SquareIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-square"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "3",
          "y": "3",
          "width": "18",
          "height": "18",
          "rx": "2",
          "ry": "2"
        })]); };
      }

    };

    var StarIcon = {
      name: 'StarIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-star"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "12 2 15.09 8.26 22 9.27 17 14.14 18.18 21.02 12 17.77 5.82 21.02 7 14.14 2 9.27 8.91 8.26 12 2"
        })]); };
      }

    };

    var StopCircleIcon = {
      name: 'StopCircleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-stop-circle"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("rect", {
          "x": "9",
          "y": "9",
          "width": "6",
          "height": "6"
        })]); };
      }

    };

    var SunIcon = {
      name: 'SunIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-sun"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "5"
        }), createVNode("line", {
          "x1": "12",
          "y1": "1",
          "x2": "12",
          "y2": "3"
        }), createVNode("line", {
          "x1": "12",
          "y1": "21",
          "x2": "12",
          "y2": "23"
        }), createVNode("line", {
          "x1": "4.22",
          "y1": "4.22",
          "x2": "5.64",
          "y2": "5.64"
        }), createVNode("line", {
          "x1": "18.36",
          "y1": "18.36",
          "x2": "19.78",
          "y2": "19.78"
        }), createVNode("line", {
          "x1": "1",
          "y1": "12",
          "x2": "3",
          "y2": "12"
        }), createVNode("line", {
          "x1": "21",
          "y1": "12",
          "x2": "23",
          "y2": "12"
        }), createVNode("line", {
          "x1": "4.22",
          "y1": "19.78",
          "x2": "5.64",
          "y2": "18.36"
        }), createVNode("line", {
          "x1": "18.36",
          "y1": "5.64",
          "x2": "19.78",
          "y2": "4.22"
        })]); };
      }

    };

    var SunriseIcon = {
      name: 'SunriseIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-sunrise"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M17 18a5 5 0 0 0-10 0"
        }), createVNode("line", {
          "x1": "12",
          "y1": "2",
          "x2": "12",
          "y2": "9"
        }), createVNode("line", {
          "x1": "4.22",
          "y1": "10.22",
          "x2": "5.64",
          "y2": "11.64"
        }), createVNode("line", {
          "x1": "1",
          "y1": "18",
          "x2": "3",
          "y2": "18"
        }), createVNode("line", {
          "x1": "21",
          "y1": "18",
          "x2": "23",
          "y2": "18"
        }), createVNode("line", {
          "x1": "18.36",
          "y1": "11.64",
          "x2": "19.78",
          "y2": "10.22"
        }), createVNode("line", {
          "x1": "23",
          "y1": "22",
          "x2": "1",
          "y2": "22"
        }), createVNode("polyline", {
          "points": "8 6 12 2 16 6"
        })]); };
      }

    };

    var SunsetIcon = {
      name: 'SunsetIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-sunset"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M17 18a5 5 0 0 0-10 0"
        }), createVNode("line", {
          "x1": "12",
          "y1": "9",
          "x2": "12",
          "y2": "2"
        }), createVNode("line", {
          "x1": "4.22",
          "y1": "10.22",
          "x2": "5.64",
          "y2": "11.64"
        }), createVNode("line", {
          "x1": "1",
          "y1": "18",
          "x2": "3",
          "y2": "18"
        }), createVNode("line", {
          "x1": "21",
          "y1": "18",
          "x2": "23",
          "y2": "18"
        }), createVNode("line", {
          "x1": "18.36",
          "y1": "11.64",
          "x2": "19.78",
          "y2": "10.22"
        }), createVNode("line", {
          "x1": "23",
          "y1": "22",
          "x2": "1",
          "y2": "22"
        }), createVNode("polyline", {
          "points": "16 5 12 9 8 5"
        })]); };
      }

    };

    var TabletIcon = {
      name: 'TabletIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-tablet"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "4",
          "y": "2",
          "width": "16",
          "height": "20",
          "rx": "2",
          "ry": "2"
        }), createVNode("line", {
          "x1": "12",
          "y1": "18",
          "x2": "12.01",
          "y2": "18"
        })]); };
      }

    };

    var TagIcon = {
      name: 'TagIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-tag"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M20.59 13.41l-7.17 7.17a2 2 0 0 1-2.83 0L2 12V2h10l8.59 8.59a2 2 0 0 1 0 2.82z"
        }), createVNode("line", {
          "x1": "7",
          "y1": "7",
          "x2": "7.01",
          "y2": "7"
        })]); };
      }

    };

    var TargetIcon = {
      name: 'TargetIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-target"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "6"
        }), createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "2"
        })]); };
      }

    };

    var TerminalIcon = {
      name: 'TerminalIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-terminal"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "4 17 10 11 4 5"
        }), createVNode("line", {
          "x1": "12",
          "y1": "19",
          "x2": "20",
          "y2": "19"
        })]); };
      }

    };

    var ThermometerIcon = {
      name: 'ThermometerIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-thermometer"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M14 14.76V3.5a2.5 2.5 0 0 0-5 0v11.26a4.5 4.5 0 1 0 5 0z"
        })]); };
      }

    };

    var ThumbsDownIcon = {
      name: 'ThumbsDownIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-thumbs-down"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M10 15v4a3 3 0 0 0 3 3l4-9V2H5.72a2 2 0 0 0-2 1.7l-1.38 9a2 2 0 0 0 2 2.3zm7-13h2.67A2.31 2.31 0 0 1 22 4v7a2.31 2.31 0 0 1-2.33 2H17"
        })]); };
      }

    };

    var ThumbsUpIcon = {
      name: 'ThumbsUpIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-thumbs-up"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M14 9V5a3 3 0 0 0-3-3l-4 9v11h11.28a2 2 0 0 0 2-1.7l1.38-9a2 2 0 0 0-2-2.3zM7 22H4a2 2 0 0 1-2-2v-7a2 2 0 0 1 2-2h3"
        })]); };
      }

    };

    var ToggleLeftIcon = {
      name: 'ToggleLeftIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-toggle-left"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "1",
          "y": "5",
          "width": "22",
          "height": "14",
          "rx": "7",
          "ry": "7"
        }), createVNode("circle", {
          "cx": "8",
          "cy": "12",
          "r": "3"
        })]); };
      }

    };

    var ToggleRightIcon = {
      name: 'ToggleRightIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-toggle-right"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "1",
          "y": "5",
          "width": "22",
          "height": "14",
          "rx": "7",
          "ry": "7"
        }), createVNode("circle", {
          "cx": "16",
          "cy": "12",
          "r": "3"
        })]); };
      }

    };

    var ToolIcon = {
      name: 'ToolIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-tool"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M14.7 6.3a1 1 0 0 0 0 1.4l1.6 1.6a1 1 0 0 0 1.4 0l3.77-3.77a6 6 0 0 1-7.94 7.94l-6.91 6.91a2.12 2.12 0 0 1-3-3l6.91-6.91a6 6 0 0 1 7.94-7.94l-3.76 3.76z"
        })]); };
      }

    };

    var Trash2Icon = {
      name: 'Trash2Icon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-trash-2"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "3 6 5 6 21 6"
        }), createVNode("path", {
          "d": "M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"
        }), createVNode("line", {
          "x1": "10",
          "y1": "11",
          "x2": "10",
          "y2": "17"
        }), createVNode("line", {
          "x1": "14",
          "y1": "11",
          "x2": "14",
          "y2": "17"
        })]); };
      }

    };

    var TrashIcon = {
      name: 'TrashIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-trash"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "3 6 5 6 21 6"
        }), createVNode("path", {
          "d": "M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"
        })]); };
      }

    };

    var TrelloIcon = {
      name: 'TrelloIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-trello"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "3",
          "y": "3",
          "width": "18",
          "height": "18",
          "rx": "2",
          "ry": "2"
        }), createVNode("rect", {
          "x": "7",
          "y": "7",
          "width": "3",
          "height": "9"
        }), createVNode("rect", {
          "x": "14",
          "y": "7",
          "width": "3",
          "height": "5"
        })]); };
      }

    };

    var TrendingDownIcon = {
      name: 'TrendingDownIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-trending-down"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "23 18 13.5 8.5 8.5 13.5 1 6"
        }), createVNode("polyline", {
          "points": "17 18 23 18 23 12"
        })]); };
      }

    };

    var TrendingUpIcon = {
      name: 'TrendingUpIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-trending-up"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "23 6 13.5 15.5 8.5 10.5 1 18"
        }), createVNode("polyline", {
          "points": "17 6 23 6 23 12"
        })]); };
      }

    };

    var TriangleIcon = {
      name: 'TriangleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-triangle"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"
        })]); };
      }

    };

    var TruckIcon = {
      name: 'TruckIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-truck"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "1",
          "y": "3",
          "width": "15",
          "height": "13"
        }), createVNode("polygon", {
          "points": "16 8 20 8 23 11 23 16 16 16 16 8"
        }), createVNode("circle", {
          "cx": "5.5",
          "cy": "18.5",
          "r": "2.5"
        }), createVNode("circle", {
          "cx": "18.5",
          "cy": "18.5",
          "r": "2.5"
        })]); };
      }

    };

    var TvIcon = {
      name: 'TvIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-tv"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "2",
          "y": "7",
          "width": "20",
          "height": "15",
          "rx": "2",
          "ry": "2"
        }), createVNode("polyline", {
          "points": "17 2 12 7 7 2"
        })]); };
      }

    };

    var TwitchIcon = {
      name: 'TwitchIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-twitch"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M21 2H3v16h5v4l4-4h5l4-4V2zm-10 9V7m5 4V7"
        })]); };
      }

    };

    var TwitterIcon = {
      name: 'TwitterIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-twitter"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M23 3a10.9 10.9 0 0 1-3.14 1.53 4.48 4.48 0 0 0-7.86 3v1A10.66 10.66 0 0 1 3 4s-4 9 5 13a11.64 11.64 0 0 1-7 2c9 5 20 0 20-11.5a4.5 4.5 0 0 0-.08-.83A7.72 7.72 0 0 0 23 3z"
        })]); };
      }

    };

    var TypeIcon = {
      name: 'TypeIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-type"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "4 7 4 4 20 4 20 7"
        }), createVNode("line", {
          "x1": "9",
          "y1": "20",
          "x2": "15",
          "y2": "20"
        }), createVNode("line", {
          "x1": "12",
          "y1": "4",
          "x2": "12",
          "y2": "20"
        })]); };
      }

    };

    var UmbrellaIcon = {
      name: 'UmbrellaIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-umbrella"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M23 12a11.05 11.05 0 0 0-22 0zm-5 7a3 3 0 0 1-6 0v-7"
        })]); };
      }

    };

    var UnderlineIcon = {
      name: 'UnderlineIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-underline"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M6 3v7a6 6 0 0 0 6 6 6 6 0 0 0 6-6V3"
        }), createVNode("line", {
          "x1": "4",
          "y1": "21",
          "x2": "20",
          "y2": "21"
        })]); };
      }

    };

    var UnlockIcon = {
      name: 'UnlockIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-unlock"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "3",
          "y": "11",
          "width": "18",
          "height": "11",
          "rx": "2",
          "ry": "2"
        }), createVNode("path", {
          "d": "M7 11V7a5 5 0 0 1 9.9-1"
        })]); };
      }

    };

    var UploadCloudIcon = {
      name: 'UploadCloudIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-upload-cloud"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "16 16 12 12 8 16"
        }), createVNode("line", {
          "x1": "12",
          "y1": "12",
          "x2": "12",
          "y2": "21"
        }), createVNode("path", {
          "d": "M20.39 18.39A5 5 0 0 0 18 9h-1.26A8 8 0 1 0 3 16.3"
        }), createVNode("polyline", {
          "points": "16 16 12 12 8 16"
        })]); };
      }

    };

    var UploadIcon = {
      name: 'UploadIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-upload"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"
        }), createVNode("polyline", {
          "points": "17 8 12 3 7 8"
        }), createVNode("line", {
          "x1": "12",
          "y1": "3",
          "x2": "12",
          "y2": "15"
        })]); };
      }

    };

    var UserCheckIcon = {
      name: 'UserCheckIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-user-check"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M16 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"
        }), createVNode("circle", {
          "cx": "8.5",
          "cy": "7",
          "r": "4"
        }), createVNode("polyline", {
          "points": "17 11 19 13 23 9"
        })]); };
      }

    };

    var UserMinusIcon = {
      name: 'UserMinusIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-user-minus"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M16 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"
        }), createVNode("circle", {
          "cx": "8.5",
          "cy": "7",
          "r": "4"
        }), createVNode("line", {
          "x1": "23",
          "y1": "11",
          "x2": "17",
          "y2": "11"
        })]); };
      }

    };

    var UserPlusIcon = {
      name: 'UserPlusIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-user-plus"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M16 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"
        }), createVNode("circle", {
          "cx": "8.5",
          "cy": "7",
          "r": "4"
        }), createVNode("line", {
          "x1": "20",
          "y1": "8",
          "x2": "20",
          "y2": "14"
        }), createVNode("line", {
          "x1": "23",
          "y1": "11",
          "x2": "17",
          "y2": "11"
        })]); };
      }

    };

    var UserXIcon = {
      name: 'UserXIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-user-x"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M16 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"
        }), createVNode("circle", {
          "cx": "8.5",
          "cy": "7",
          "r": "4"
        }), createVNode("line", {
          "x1": "18",
          "y1": "8",
          "x2": "23",
          "y2": "13"
        }), createVNode("line", {
          "x1": "23",
          "y1": "8",
          "x2": "18",
          "y2": "13"
        })]); };
      }

    };

    var UserIcon = {
      name: 'UserIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-user"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"
        }), createVNode("circle", {
          "cx": "12",
          "cy": "7",
          "r": "4"
        })]); };
      }

    };

    var UsersIcon = {
      name: 'UsersIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-users"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"
        }), createVNode("circle", {
          "cx": "9",
          "cy": "7",
          "r": "4"
        }), createVNode("path", {
          "d": "M23 21v-2a4 4 0 0 0-3-3.87"
        }), createVNode("path", {
          "d": "M16 3.13a4 4 0 0 1 0 7.75"
        })]); };
      }

    };

    var VideoOffIcon = {
      name: 'VideoOffIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-video-off"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M16 16v1a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V7a2 2 0 0 1 2-2h2m5.66 0H14a2 2 0 0 1 2 2v3.34l1 1L23 7v10"
        }), createVNode("line", {
          "x1": "1",
          "y1": "1",
          "x2": "23",
          "y2": "23"
        })]); };
      }

    };

    var VideoIcon = {
      name: 'VideoIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-video"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "23 7 16 12 23 17 23 7"
        }), createVNode("rect", {
          "x": "1",
          "y": "5",
          "width": "15",
          "height": "14",
          "rx": "2",
          "ry": "2"
        })]); };
      }

    };

    var VoicemailIcon = {
      name: 'VoicemailIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-voicemail"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "5.5",
          "cy": "11.5",
          "r": "4.5"
        }), createVNode("circle", {
          "cx": "18.5",
          "cy": "11.5",
          "r": "4.5"
        }), createVNode("line", {
          "x1": "5.5",
          "y1": "16",
          "x2": "18.5",
          "y2": "16"
        })]); };
      }

    };

    var Volume1Icon = {
      name: 'Volume1Icon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-volume-1"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "11 5 6 9 2 9 2 15 6 15 11 19 11 5"
        }), createVNode("path", {
          "d": "M15.54 8.46a5 5 0 0 1 0 7.07"
        })]); };
      }

    };

    var Volume2Icon = {
      name: 'Volume2Icon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-volume-2"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "11 5 6 9 2 9 2 15 6 15 11 19 11 5"
        }), createVNode("path", {
          "d": "M19.07 4.93a10 10 0 0 1 0 14.14M15.54 8.46a5 5 0 0 1 0 7.07"
        })]); };
      }

    };

    var VolumeXIcon = {
      name: 'VolumeXIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-volume-x"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "11 5 6 9 2 9 2 15 6 15 11 19 11 5"
        }), createVNode("line", {
          "x1": "23",
          "y1": "9",
          "x2": "17",
          "y2": "15"
        }), createVNode("line", {
          "x1": "17",
          "y1": "9",
          "x2": "23",
          "y2": "15"
        })]); };
      }

    };

    var VolumeIcon = {
      name: 'VolumeIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-volume"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "11 5 6 9 2 9 2 15 6 15 11 19 11 5"
        })]); };
      }

    };

    var WatchIcon = {
      name: 'WatchIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-watch"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "7"
        }), createVNode("polyline", {
          "points": "12 9 12 12 13.5 13.5"
        }), createVNode("path", {
          "d": "M16.51 17.35l-.35 3.83a2 2 0 0 1-2 1.82H9.83a2 2 0 0 1-2-1.82l-.35-3.83m.01-10.7l.35-3.83A2 2 0 0 1 9.83 1h4.35a2 2 0 0 1 2 1.82l.35 3.83"
        })]); };
      }

    };

    var WifiOffIcon = {
      name: 'WifiOffIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-wifi-off"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "1",
          "y1": "1",
          "x2": "23",
          "y2": "23"
        }), createVNode("path", {
          "d": "M16.72 11.06A10.94 10.94 0 0 1 19 12.55"
        }), createVNode("path", {
          "d": "M5 12.55a10.94 10.94 0 0 1 5.17-2.39"
        }), createVNode("path", {
          "d": "M10.71 5.05A16 16 0 0 1 22.58 9"
        }), createVNode("path", {
          "d": "M1.42 9a15.91 15.91 0 0 1 4.7-2.88"
        }), createVNode("path", {
          "d": "M8.53 16.11a6 6 0 0 1 6.95 0"
        }), createVNode("line", {
          "x1": "12",
          "y1": "20",
          "x2": "12.01",
          "y2": "20"
        })]); };
      }

    };

    var WifiIcon = {
      name: 'WifiIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-wifi"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M5 12.55a11 11 0 0 1 14.08 0"
        }), createVNode("path", {
          "d": "M1.42 9a16 16 0 0 1 21.16 0"
        }), createVNode("path", {
          "d": "M8.53 16.11a6 6 0 0 1 6.95 0"
        }), createVNode("line", {
          "x1": "12",
          "y1": "20",
          "x2": "12.01",
          "y2": "20"
        })]); };
      }

    };

    var WindIcon = {
      name: 'WindIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-wind"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M9.59 4.59A2 2 0 1 1 11 8H2m10.59 11.41A2 2 0 1 0 14 16H2m15.73-8.27A2.5 2.5 0 1 1 19.5 12H2"
        })]); };
      }

    };

    var XCircleIcon = {
      name: 'XCircleIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-x-circle"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "12",
          "cy": "12",
          "r": "10"
        }), createVNode("line", {
          "x1": "15",
          "y1": "9",
          "x2": "9",
          "y2": "15"
        }), createVNode("line", {
          "x1": "9",
          "y1": "9",
          "x2": "15",
          "y2": "15"
        })]); };
      }

    };

    var XOctagonIcon = {
      name: 'XOctagonIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-x-octagon"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "7.86 2 16.14 2 22 7.86 22 16.14 16.14 22 7.86 22 2 16.14 2 7.86 7.86 2"
        }), createVNode("line", {
          "x1": "15",
          "y1": "9",
          "x2": "9",
          "y2": "15"
        }), createVNode("line", {
          "x1": "9",
          "y1": "9",
          "x2": "15",
          "y2": "15"
        })]); };
      }

    };

    var XSquareIcon = {
      name: 'XSquareIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-x-square"
        }, Object.assign({}, attrs)), [createVNode("rect", {
          "x": "3",
          "y": "3",
          "width": "18",
          "height": "18",
          "rx": "2",
          "ry": "2"
        }), createVNode("line", {
          "x1": "9",
          "y1": "9",
          "x2": "15",
          "y2": "15"
        }), createVNode("line", {
          "x1": "15",
          "y1": "9",
          "x2": "9",
          "y2": "15"
        })]); };
      }

    };

    var XIcon = {
      name: 'XIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-x"
        }, Object.assign({}, attrs)), [createVNode("line", {
          "x1": "18",
          "y1": "6",
          "x2": "6",
          "y2": "18"
        }), createVNode("line", {
          "x1": "6",
          "y1": "6",
          "x2": "18",
          "y2": "18"
        })]); };
      }

    };

    var YoutubeIcon = {
      name: 'YoutubeIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-youtube"
        }, Object.assign({}, attrs)), [createVNode("path", {
          "d": "M22.54 6.42a2.78 2.78 0 0 0-1.94-2C18.88 4 12 4 12 4s-6.88 0-8.6.46a2.78 2.78 0 0 0-1.94 2A29 29 0 0 0 1 11.75a29 29 0 0 0 .46 5.33A2.78 2.78 0 0 0 3.4 19c1.72.46 8.6.46 8.6.46s6.88 0 8.6-.46a2.78 2.78 0 0 0 1.94-2 29 29 0 0 0 .46-5.25 29 29 0 0 0-.46-5.33z"
        }), createVNode("polygon", {
          "points": "9.75 15.02 15.5 11.75 9.75 8.48 9.75 15.02"
        })]); };
      }

    };

    var ZapOffIcon = {
      name: 'ZapOffIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-zap-off"
        }, Object.assign({}, attrs)), [createVNode("polyline", {
          "points": "12.41 6.75 13 2 10.57 4.92"
        }), createVNode("polyline", {
          "points": "18.57 12.91 21 10 15.66 10"
        }), createVNode("polyline", {
          "points": "8 8 3 14 12 14 11 22 16 16"
        }), createVNode("line", {
          "x1": "1",
          "y1": "1",
          "x2": "23",
          "y2": "23"
        })]); };
      }

    };

    var ZapIcon = {
      name: 'ZapIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-zap"
        }, Object.assign({}, attrs)), [createVNode("polygon", {
          "points": "13 2 3 14 12 14 11 22 21 10 12 10 13 2"
        })]); };
      }

    };

    var ZoomInIcon = {
      name: 'ZoomInIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-zoom-in"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "11",
          "cy": "11",
          "r": "8"
        }), createVNode("line", {
          "x1": "21",
          "y1": "21",
          "x2": "16.65",
          "y2": "16.65"
        }), createVNode("line", {
          "x1": "11",
          "y1": "8",
          "x2": "11",
          "y2": "14"
        }), createVNode("line", {
          "x1": "8",
          "y1": "11",
          "x2": "14",
          "y2": "11"
        })]); };
      }

    };

    var ZoomOutIcon = {
      name: 'ZoomOutIcon',
      props: {
        size: {
          type: String,
          default: '16',
          validator: function (s) { return !isNaN(s) || s.length >= 2 && !isNaN(s.slice(0, s.length - 1)) && s.slice(-1) === 'x'; }
        }
      },

      setup: function setup(props, ctx) {
        var size = props.size.slice(-1) === 'x' ? props.size.slice(0, props.size.length - 1) + 'em' : parseInt(props.size) + 'px';
        var attrs = Object.assign({}, ctx.attrs);
        attrs.width = attrs.width || size;
        attrs.height = attrs.height || size;
        return function () { return createVNode("svg", mergeProps({
          "xmlns": "http://www.w3.org/2000/svg",
          "width": "24",
          "height": "24",
          "viewBox": "0 0 24 24",
          "fill": "none",
          "stroke": "currentColor",
          "stroke-width": "2",
          "stroke-linecap": "round",
          "stroke-linejoin": "round",
          "class": "feather feather-zoom-out"
        }, Object.assign({}, attrs)), [createVNode("circle", {
          "cx": "11",
          "cy": "11",
          "r": "8"
        }), createVNode("line", {
          "x1": "21",
          "y1": "21",
          "x2": "16.65",
          "y2": "16.65"
        }), createVNode("line", {
          "x1": "8",
          "y1": "11",
          "x2": "14",
          "y2": "11"
        })]); };
      }

    };

    exports.ActivityIcon = ActivityIcon;
    exports.AirplayIcon = AirplayIcon;
    exports.AlertCircleIcon = AlertCircleIcon;
    exports.AlertOctagonIcon = AlertOctagonIcon;
    exports.AlertTriangleIcon = AlertTriangleIcon;
    exports.AlignCenterIcon = AlignCenterIcon;
    exports.AlignJustifyIcon = AlignJustifyIcon;
    exports.AlignLeftIcon = AlignLeftIcon;
    exports.AlignRightIcon = AlignRightIcon;
    exports.AnchorIcon = AnchorIcon;
    exports.ApertureIcon = ApertureIcon;
    exports.ArchiveIcon = ArchiveIcon;
    exports.ArrowDownCircleIcon = ArrowDownCircleIcon;
    exports.ArrowDownLeftIcon = ArrowDownLeftIcon;
    exports.ArrowDownRightIcon = ArrowDownRightIcon;
    exports.ArrowDownIcon = ArrowDownIcon;
    exports.ArrowLeftCircleIcon = ArrowLeftCircleIcon;
    exports.ArrowLeftIcon = ArrowLeftIcon;
    exports.ArrowRightCircleIcon = ArrowRightCircleIcon;
    exports.ArrowRightIcon = ArrowRightIcon;
    exports.ArrowUpCircleIcon = ArrowUpCircleIcon;
    exports.ArrowUpLeftIcon = ArrowUpLeftIcon;
    exports.ArrowUpRightIcon = ArrowUpRightIcon;
    exports.ArrowUpIcon = ArrowUpIcon;
    exports.AtSignIcon = AtSignIcon;
    exports.AwardIcon = AwardIcon;
    exports.BarChart2Icon = BarChart2Icon;
    exports.BarChartIcon = BarChartIcon;
    exports.BatteryChargingIcon = BatteryChargingIcon;
    exports.BatteryIcon = BatteryIcon;
    exports.BellOffIcon = BellOffIcon;
    exports.BellIcon = BellIcon;
    exports.BluetoothIcon = BluetoothIcon;
    exports.BoldIcon = BoldIcon;
    exports.BookOpenIcon = BookOpenIcon;
    exports.BookIcon = BookIcon;
    exports.BookmarkIcon = BookmarkIcon;
    exports.BoxIcon = BoxIcon;
    exports.BriefcaseIcon = BriefcaseIcon;
    exports.CalendarIcon = CalendarIcon;
    exports.CameraOffIcon = CameraOffIcon;
    exports.CameraIcon = CameraIcon;
    exports.CastIcon = CastIcon;
    exports.CheckCircleIcon = CheckCircleIcon;
    exports.CheckSquareIcon = CheckSquareIcon;
    exports.CheckIcon = CheckIcon;
    exports.ChevronDownIcon = ChevronDownIcon;
    exports.ChevronLeftIcon = ChevronLeftIcon;
    exports.ChevronRightIcon = ChevronRightIcon;
    exports.ChevronUpIcon = ChevronUpIcon;
    exports.ChevronsDownIcon = ChevronsDownIcon;
    exports.ChevronsLeftIcon = ChevronsLeftIcon;
    exports.ChevronsRightIcon = ChevronsRightIcon;
    exports.ChevronsUpIcon = ChevronsUpIcon;
    exports.ChromeIcon = ChromeIcon;
    exports.CircleIcon = CircleIcon;
    exports.ClipboardIcon = ClipboardIcon;
    exports.ClockIcon = ClockIcon;
    exports.CloudDrizzleIcon = CloudDrizzleIcon;
    exports.CloudLightningIcon = CloudLightningIcon;
    exports.CloudOffIcon = CloudOffIcon;
    exports.CloudRainIcon = CloudRainIcon;
    exports.CloudSnowIcon = CloudSnowIcon;
    exports.CloudIcon = CloudIcon;
    exports.CodeIcon = CodeIcon;
    exports.CodepenIcon = CodepenIcon;
    exports.CodesandboxIcon = CodesandboxIcon;
    exports.CoffeeIcon = CoffeeIcon;
    exports.ColumnsIcon = ColumnsIcon;
    exports.CommandIcon = CommandIcon;
    exports.CompassIcon = CompassIcon;
    exports.CopyIcon = CopyIcon;
    exports.CornerDownLeftIcon = CornerDownLeftIcon;
    exports.CornerDownRightIcon = CornerDownRightIcon;
    exports.CornerLeftDownIcon = CornerLeftDownIcon;
    exports.CornerLeftUpIcon = CornerLeftUpIcon;
    exports.CornerRightDownIcon = CornerRightDownIcon;
    exports.CornerRightUpIcon = CornerRightUpIcon;
    exports.CornerUpLeftIcon = CornerUpLeftIcon;
    exports.CornerUpRightIcon = CornerUpRightIcon;
    exports.CpuIcon = CpuIcon;
    exports.CreditCardIcon = CreditCardIcon;
    exports.CropIcon = CropIcon;
    exports.CrosshairIcon = CrosshairIcon;
    exports.DatabaseIcon = DatabaseIcon;
    exports.DeleteIcon = DeleteIcon;
    exports.DiscIcon = DiscIcon;
    exports.DivideCircleIcon = DivideCircleIcon;
    exports.DivideSquareIcon = DivideSquareIcon;
    exports.DivideIcon = DivideIcon;
    exports.DollarSignIcon = DollarSignIcon;
    exports.DownloadCloudIcon = DownloadCloudIcon;
    exports.DownloadIcon = DownloadIcon;
    exports.DribbbleIcon = DribbbleIcon;
    exports.DropletIcon = DropletIcon;
    exports.Edit2Icon = Edit2Icon;
    exports.Edit3Icon = Edit3Icon;
    exports.EditIcon = EditIcon;
    exports.ExternalLinkIcon = ExternalLinkIcon;
    exports.EyeOffIcon = EyeOffIcon;
    exports.EyeIcon = EyeIcon;
    exports.FacebookIcon = FacebookIcon;
    exports.FastForwardIcon = FastForwardIcon;
    exports.FeatherIcon = FeatherIcon;
    exports.FigmaIcon = FigmaIcon;
    exports.FileMinusIcon = FileMinusIcon;
    exports.FilePlusIcon = FilePlusIcon;
    exports.FileTextIcon = FileTextIcon;
    exports.FileIcon = FileIcon;
    exports.FilmIcon = FilmIcon;
    exports.FilterIcon = FilterIcon;
    exports.FlagIcon = FlagIcon;
    exports.FolderMinusIcon = FolderMinusIcon;
    exports.FolderPlusIcon = FolderPlusIcon;
    exports.FolderIcon = FolderIcon;
    exports.FramerIcon = FramerIcon;
    exports.FrownIcon = FrownIcon;
    exports.GiftIcon = GiftIcon;
    exports.GitBranchIcon = GitBranchIcon;
    exports.GitCommitIcon = GitCommitIcon;
    exports.GitMergeIcon = GitMergeIcon;
    exports.GitPullRequestIcon = GitPullRequestIcon;
    exports.GithubIcon = GithubIcon;
    exports.GitlabIcon = GitlabIcon;
    exports.GlobeIcon = GlobeIcon;
    exports.GridIcon = GridIcon;
    exports.HardDriveIcon = HardDriveIcon;
    exports.HashIcon = HashIcon;
    exports.HeadphonesIcon = HeadphonesIcon;
    exports.HeartIcon = HeartIcon;
    exports.HelpCircleIcon = HelpCircleIcon;
    exports.HexagonIcon = HexagonIcon;
    exports.HomeIcon = HomeIcon;
    exports.ImageIcon = ImageIcon;
    exports.InboxIcon = InboxIcon;
    exports.InfoIcon = InfoIcon;
    exports.InstagramIcon = InstagramIcon;
    exports.ItalicIcon = ItalicIcon;
    exports.KeyIcon = KeyIcon;
    exports.LayersIcon = LayersIcon;
    exports.LayoutIcon = LayoutIcon;
    exports.LifeBuoyIcon = LifeBuoyIcon;
    exports.Link2Icon = Link2Icon;
    exports.LinkIcon = LinkIcon;
    exports.LinkedinIcon = LinkedinIcon;
    exports.ListIcon = ListIcon;
    exports.LoaderIcon = LoaderIcon;
    exports.LockIcon = LockIcon;
    exports.LogInIcon = LogInIcon;
    exports.LogOutIcon = LogOutIcon;
    exports.MailIcon = MailIcon;
    exports.MapPinIcon = MapPinIcon;
    exports.MapIcon = MapIcon;
    exports.Maximize2Icon = Maximize2Icon;
    exports.MaximizeIcon = MaximizeIcon;
    exports.MehIcon = MehIcon;
    exports.MenuIcon = MenuIcon;
    exports.MessageCircleIcon = MessageCircleIcon;
    exports.MessageSquareIcon = MessageSquareIcon;
    exports.MicOffIcon = MicOffIcon;
    exports.MicIcon = MicIcon;
    exports.Minimize2Icon = Minimize2Icon;
    exports.MinimizeIcon = MinimizeIcon;
    exports.MinusCircleIcon = MinusCircleIcon;
    exports.MinusSquareIcon = MinusSquareIcon;
    exports.MinusIcon = MinusIcon;
    exports.MonitorIcon = MonitorIcon;
    exports.MoonIcon = MoonIcon;
    exports.MoreHorizontalIcon = MoreHorizontalIcon;
    exports.MoreVerticalIcon = MoreVerticalIcon;
    exports.MousePointerIcon = MousePointerIcon;
    exports.MoveIcon = MoveIcon;
    exports.MusicIcon = MusicIcon;
    exports.Navigation2Icon = Navigation2Icon;
    exports.NavigationIcon = NavigationIcon;
    exports.OctagonIcon = OctagonIcon;
    exports.PackageIcon = PackageIcon;
    exports.PaperclipIcon = PaperclipIcon;
    exports.PauseCircleIcon = PauseCircleIcon;
    exports.PauseIcon = PauseIcon;
    exports.PenToolIcon = PenToolIcon;
    exports.PercentIcon = PercentIcon;
    exports.PhoneCallIcon = PhoneCallIcon;
    exports.PhoneForwardedIcon = PhoneForwardedIcon;
    exports.PhoneIncomingIcon = PhoneIncomingIcon;
    exports.PhoneMissedIcon = PhoneMissedIcon;
    exports.PhoneOffIcon = PhoneOffIcon;
    exports.PhoneOutgoingIcon = PhoneOutgoingIcon;
    exports.PhoneIcon = PhoneIcon;
    exports.PieChartIcon = PieChartIcon;
    exports.PlayCircleIcon = PlayCircleIcon;
    exports.PlayIcon = PlayIcon;
    exports.PlusCircleIcon = PlusCircleIcon;
    exports.PlusSquareIcon = PlusSquareIcon;
    exports.PlusIcon = PlusIcon;
    exports.PocketIcon = PocketIcon;
    exports.PowerIcon = PowerIcon;
    exports.PrinterIcon = PrinterIcon;
    exports.RadioIcon = RadioIcon;
    exports.RefreshCcwIcon = RefreshCcwIcon;
    exports.RefreshCwIcon = RefreshCwIcon;
    exports.RepeatIcon = RepeatIcon;
    exports.RewindIcon = RewindIcon;
    exports.RotateCcwIcon = RotateCcwIcon;
    exports.RotateCwIcon = RotateCwIcon;
    exports.RssIcon = RssIcon;
    exports.SaveIcon = SaveIcon;
    exports.ScissorsIcon = ScissorsIcon;
    exports.SearchIcon = SearchIcon;
    exports.SendIcon = SendIcon;
    exports.ServerIcon = ServerIcon;
    exports.SettingsIcon = SettingsIcon;
    exports.Share2Icon = Share2Icon;
    exports.ShareIcon = ShareIcon;
    exports.ShieldOffIcon = ShieldOffIcon;
    exports.ShieldIcon = ShieldIcon;
    exports.ShoppingBagIcon = ShoppingBagIcon;
    exports.ShoppingCartIcon = ShoppingCartIcon;
    exports.ShuffleIcon = ShuffleIcon;
    exports.SidebarIcon = SidebarIcon;
    exports.SkipBackIcon = SkipBackIcon;
    exports.SkipForwardIcon = SkipForwardIcon;
    exports.SlackIcon = SlackIcon;
    exports.SlashIcon = SlashIcon;
    exports.SlidersIcon = SlidersIcon;
    exports.SmartphoneIcon = SmartphoneIcon;
    exports.SmileIcon = SmileIcon;
    exports.SpeakerIcon = SpeakerIcon;
    exports.SquareIcon = SquareIcon;
    exports.StarIcon = StarIcon;
    exports.StopCircleIcon = StopCircleIcon;
    exports.SunIcon = SunIcon;
    exports.SunriseIcon = SunriseIcon;
    exports.SunsetIcon = SunsetIcon;
    exports.TabletIcon = TabletIcon;
    exports.TagIcon = TagIcon;
    exports.TargetIcon = TargetIcon;
    exports.TerminalIcon = TerminalIcon;
    exports.ThermometerIcon = ThermometerIcon;
    exports.ThumbsDownIcon = ThumbsDownIcon;
    exports.ThumbsUpIcon = ThumbsUpIcon;
    exports.ToggleLeftIcon = ToggleLeftIcon;
    exports.ToggleRightIcon = ToggleRightIcon;
    exports.ToolIcon = ToolIcon;
    exports.Trash2Icon = Trash2Icon;
    exports.TrashIcon = TrashIcon;
    exports.TrelloIcon = TrelloIcon;
    exports.TrendingDownIcon = TrendingDownIcon;
    exports.TrendingUpIcon = TrendingUpIcon;
    exports.TriangleIcon = TriangleIcon;
    exports.TruckIcon = TruckIcon;
    exports.TvIcon = TvIcon;
    exports.TwitchIcon = TwitchIcon;
    exports.TwitterIcon = TwitterIcon;
    exports.TypeIcon = TypeIcon;
    exports.UmbrellaIcon = UmbrellaIcon;
    exports.UnderlineIcon = UnderlineIcon;
    exports.UnlockIcon = UnlockIcon;
    exports.UploadCloudIcon = UploadCloudIcon;
    exports.UploadIcon = UploadIcon;
    exports.UserCheckIcon = UserCheckIcon;
    exports.UserMinusIcon = UserMinusIcon;
    exports.UserPlusIcon = UserPlusIcon;
    exports.UserXIcon = UserXIcon;
    exports.UserIcon = UserIcon;
    exports.UsersIcon = UsersIcon;
    exports.VideoOffIcon = VideoOffIcon;
    exports.VideoIcon = VideoIcon;
    exports.VoicemailIcon = VoicemailIcon;
    exports.Volume1Icon = Volume1Icon;
    exports.Volume2Icon = Volume2Icon;
    exports.VolumeXIcon = VolumeXIcon;
    exports.VolumeIcon = VolumeIcon;
    exports.WatchIcon = WatchIcon;
    exports.WifiOffIcon = WifiOffIcon;
    exports.WifiIcon = WifiIcon;
    exports.WindIcon = WindIcon;
    exports.XCircleIcon = XCircleIcon;
    exports.XOctagonIcon = XOctagonIcon;
    exports.XSquareIcon = XSquareIcon;
    exports.XIcon = XIcon;
    exports.YoutubeIcon = YoutubeIcon;
    exports.ZapOffIcon = ZapOffIcon;
    exports.ZapIcon = ZapIcon;
    exports.ZoomInIcon = ZoomInIcon;
    exports.ZoomOutIcon = ZoomOutIcon;

    Object.defineProperty(exports, '__esModule', { value: true });

})));
