/**
 * @license Copyright (c) 2003-2023, CKSource Holding sp. z o.o. All rights reserved.
 * For licensing, see LICENSE.md or https://ckeditor.com/legal/ckeditor-oss-license
 */
/**
 * @module image/imageinsert/imageinsertviaurlui
 */
import { icons, Plugin } from 'ckeditor5/src/core';
import { ButtonView, CollapsibleView, DropdownButtonView } from 'ckeditor5/src/ui';
import ImageInsertUI from './imageinsertui';
import ImageInsertUrlView from './ui/imageinserturlview';
/**
 * The image insert via URL plugin (UI part).
 *
 * For a detailed overview, check the {@glink features/images/images-inserting
 * Insert images via source URL} documentation.
 *
 * This plugin registers the {@link module:image/imageinsert/imageinsertui~ImageInsertUI} integration for `url`.
 */
export default class ImageInsertViaUrlUI extends Plugin {
    /**
     * @inheritDoc
     */
    static get pluginName() {
        return 'ImageInsertViaUrlUI';
    }
    /**
     * @inheritDoc
     */
    static get requires() {
        return [ImageInsertUI];
    }
    /**
     * @inheritDoc
     */
    init() {
        this._imageInsertUI = this.editor.plugins.get('ImageInsertUI');
        const insertImageCommand = this.editor.commands.get('insertImage');
        this._imageInsertUI.registerIntegration({
            name: 'url',
            observable: insertImageCommand,
            requiresForm: true,
            buttonViewCreator: isOnlyOne => this._createInsertUrlButton(isOnlyOne),
            formViewCreator: isOnlyOne => this._createInsertUrlView(isOnlyOne)
        });
    }
    /**
     * Creates the view displayed in the dropdown.
     */
    _createInsertUrlView(isOnlyOne) {
        const editor = this.editor;
        const locale = editor.locale;
        const t = locale.t;
        const replaceImageSourceCommand = editor.commands.get('replaceImageSource');
        const insertImageCommand = editor.commands.get('insertImage');
        const imageInsertUrlView = new ImageInsertUrlView(locale);
        const collapsibleView = isOnlyOne ? null : new CollapsibleView(locale, [imageInsertUrlView]);
        imageInsertUrlView.bind('isImageSelected').to(this._imageInsertUI);
        imageInsertUrlView.bind('isEnabled').toMany([insertImageCommand, replaceImageSourceCommand], 'isEnabled', (...isEnabled) => (isEnabled.some(isCommandEnabled => isCommandEnabled)));
        // Set initial value because integrations are created on first dropdown open.
        imageInsertUrlView.imageURLInputValue = replaceImageSourceCommand.value || '';
        this._imageInsertUI.dropdownView.on('change:isOpen', () => {
            if (this._imageInsertUI.dropdownView.isOpen) {
                // Make sure that each time the panel shows up, the URL field remains in sync with the value of
                // the command. If the user typed in the input, then canceled and re-opened it without changing
                // the value of the media command (e.g. because they didn't change the selection), they would see
                // the old value instead of the actual value of the command.
                imageInsertUrlView.imageURLInputValue = replaceImageSourceCommand.value || '';
                if (collapsibleView) {
                    collapsibleView.isCollapsed = true;
                }
            }
            // Note: Use the low priority to make sure the following listener starts working after the
            // default action of the drop-down is executed (i.e. the panel showed up). Otherwise, the
            // invisible form/input cannot be focused/selected.
        }, { priority: 'low' });
        imageInsertUrlView.on('submit', () => {
            if (replaceImageSourceCommand.isEnabled) {
                editor.execute('replaceImageSource', { source: imageInsertUrlView.imageURLInputValue });
            }
            else {
                editor.execute('insertImage', { source: imageInsertUrlView.imageURLInputValue });
            }
            this._closePanel();
        });
        imageInsertUrlView.on('cancel', () => this._closePanel());
        if (collapsibleView) {
            collapsibleView.set({
                isCollapsed: true
            });
            collapsibleView.bind('label').to(this._imageInsertUI, 'isImageSelected', isImageSelected => isImageSelected ?
                t('Update image URL') :
                t('Insert image via URL'));
            return collapsibleView;
        }
        return imageInsertUrlView;
    }
    /**
     * Creates the toolbar button.
     */
    _createInsertUrlButton(isOnlyOne) {
        const ButtonClass = isOnlyOne ? DropdownButtonView : ButtonView;
        const editor = this.editor;
        const button = new ButtonClass(editor.locale);
        const t = editor.locale.t;
        button.set({
            icon: icons.imageUrl,
            tooltip: true
        });
        button.bind('label').to(this._imageInsertUI, 'isImageSelected', isImageSelected => isImageSelected ?
            t('Update image URL') :
            t('Insert image via URL'));
        return button;
    }
    /**
     * Closes the dropdown.
     */
    _closePanel() {
        this.editor.editing.view.focus();
        this._imageInsertUI.dropdownView.isOpen = false;
    }
}
