import { Container } from "../../core/render/Container";
import { EditableLabel } from "../../core/render/EditableLabel";
import { Rectangle } from "../../core/render/Rectangle";
import { Triangle } from "../../core/render/Triangle";
import { color } from "../../core/util/Color";
/**
 * Draws an interactive NumericStepper.
 *
 * @important
 * @since 5.14.0
 */
export class NumericStepper extends Container {
    constructor() {
        super(...arguments);
        Object.defineProperty(this, "label", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: this.children.push(EditableLabel.new(this._root, {}))
        });
        Object.defineProperty(this, "buttonsContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: this.children.push(Container.new(this._root, { themeTags: ["buttons"] }))
        });
        Object.defineProperty(this, "upButton", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: this.buttonsContainer.children.push(Triangle.new(this._root, { themeTags: ["upbutton"] }))
        });
        Object.defineProperty(this, "downButton", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: this.buttonsContainer.children.push(Triangle.new(this._root, { themeTags: ["downbutton"] }))
        });
    }
    _afterNew() {
        this.addTag("numericstepper");
        this.set("layout", this._root.horizontalLayout);
        this.label.adapters.add("text", (text) => {
            if (text) {
                text.replace(/\D/g, '');
            }
            if (text === "") {
                text = "0";
            }
            return text;
        });
        this.set("background", Rectangle.new(this._root, {
            fillOpacity: 0,
            fill: color(0xffffff)
        }));
        this.buttonsContainer.states.create("hidden", {
            opacity: 0,
            visible: true
        });
        this.events.on("pointerover", () => {
            this.buttonsContainer.set("active", true);
        });
        this.events.on("pointerout", () => {
            this.buttonsContainer.set("active", false);
        });
        this.label.on("text", (text) => {
            if (text) {
                const value = parseFloat(text);
                if (!isNaN(value)) {
                    this.set("value", value);
                }
            }
        });
        this.upButton.events.on("click", () => {
            this.set("value", this.get("value", 0) + 1);
        });
        this.downButton.events.on("click", () => {
            this.set("value", Math.max(0, this.get("value", 0) - 1));
        });
        super._afterNew();
    }
    _updateChildren() {
        super._updateChildren();
        const value = this.get("value", 0);
        const label = this.label;
        if (this.isDirty("value")) {
            label.set("text", value.toString());
        }
        if (this.isDirty("disabled")) {
            const disabled = this.get("disabled", false);
            this.set("interactive", !disabled);
            this.upButton.set("interactive", !disabled);
            this.downButton.set("interactive", !disabled);
            if (disabled) {
                this.label.set("editOn", "none");
                this.buttonsContainer.hide(0);
            }
            else {
                this.label.set("editOn", "click");
                this.buttonsContainer.show(0);
            }
        }
    }
}
Object.defineProperty(NumericStepper, "className", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: "NumericStepper"
});
Object.defineProperty(NumericStepper, "classNames", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: Container.classNames.concat([NumericStepper.className])
});
//# sourceMappingURL=NumericStepper.js.map