import { Button } from "./Button";
import * as $utils from "../util/Utils";
/**
 * Draws an interactive button, which displays a confirmation when clicked.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/common-elements/buttons/} for more info
 * @important
 * @since 5.14.0
 */
export class ConfirmButton extends Button {
    constructor() {
        super(...arguments);
        Object.defineProperty(this, "_ignoreClick", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
    }
    _afterNew() {
        super._afterNew();
        this.addTag("confirm");
        if ($utils.supports("keyboardevents")) {
            this._disposers.push($utils.addEventListener(document, "keydown", (ev) => {
                const eventKey = $utils.getEventKey(ev);
                if (this.get("active") && eventKey == "Escape") {
                    this._cancel();
                }
            }));
        }
        this._disposers.push($utils.addEventListener(document, "click", () => {
            if (!this._ignoreClick) {
                this._cancel();
            }
            this._ignoreClick = false;
        }));
        this.events.on("click", () => {
            if (!this.get("active")) {
                this._ignoreClick = true;
            }
            this._confirm();
        });
    }
    _cancel() {
        if (this.get("active")) {
            this.events.dispatch("cancelled", {
                type: "cancelled",
                target: this
            });
            this.set("active", false);
        }
    }
    _confirm() {
        if (this.get("active")) {
            this.events.dispatch("confirmed", {
                type: "confirmed",
                target: this
            });
            this.set("active", false);
            if (this.isFocus()) {
                this.setTimeout(() => {
                    this.set("active", false);
                    $utils.blur();
                }, 10);
            }
        }
    }
    _prepareChildren() {
        super._prepareChildren();
    }
}
Object.defineProperty(ConfirmButton, "className", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: "ConfirmButton"
});
Object.defineProperty(ConfirmButton, "classNames", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: Button.classNames.concat([ConfirmButton.className])
});
//# sourceMappingURL=ConfirmButton.js.map