import { CurveDefaultTheme } from "./CurveDefaultTheme";
import { XYChart } from "../xy/XYChart";
import { p50 } from "../../core/util/Percent";
import { Container } from "../../core/render/Container";
import { Graphics } from "../../core/render/Graphics";
/**
 * Base chart for a Timeline chart.
 *
 * For this chart to work, it needs curve points provided via renderer of
 * its X-axis.
 *
 * Note: it is an experimental chart type and does not support all the
 * functionality of the [[XYChart]].
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/timeline/} for more info
 * @since 5.12.0
 * @important
 */
export class CurveChart extends XYChart {
    constructor() {
        super(...arguments);
        /**
         * [[Container]] where chart elements go.
         *
         * @default Container.new()
         */
        Object.defineProperty(this, "curveContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: this.plotContainer.children.push(Container.new(this._root, { x: p50, y: p50 }))
        });
    }
    _afterNew() {
        this._defaultThemes.push(CurveDefaultTheme.new(this._root));
        super._afterNew();
        const curveContainer = this.curveContainer;
        const gridContainer = this.gridContainer;
        const topGridContainer = this.topGridContainer;
        const seriesContainer = this.seriesContainer;
        const bulletsContainer = this.bulletsContainer;
        curveContainer.children.pushAll([gridContainer, seriesContainer, topGridContainer, bulletsContainer]);
        seriesContainer.set("mask", Graphics.new(this._root, {}));
        gridContainer.set("mask", Graphics.new(this._root, {}));
        this._disposers.push(this.plotContainer.events.on("boundschanged", () => {
            this._updateMasks();
        }));
    }
    _maskGrid() {
    }
    _addCursor(cursor) {
        this.curveContainer.children.push(cursor);
    }
    // do not delete
    _updateMasks() {
        this.xAxes.each((axis) => {
            const renderer = axis.get("renderer");
            renderer._updateLayout();
        });
        this._updateMask(this.seriesContainer);
        this._updateMask(this.gridContainer);
        this.series.each((series) => {
            if (series.get("maskBullets")) {
                this._updateMask(series.bulletsContainer);
            }
            else {
                series.bulletsContainer.set("mask", undefined);
            }
        });
        this.yAxes.each((axis) => {
            axis.markDirtySize();
        });
    }
    /**
     * @ignore
     */
    _updateMask(container) {
        const mask = container.get("mask");
        if (mask) {
            const xAxis = this.xAxes.getIndex(0);
            const yAxis = this.yAxes.getIndex(0);
            if (xAxis && yAxis) {
                const renderer = xAxis.get("renderer");
                const points = renderer.getPoints(xAxis.get("start", 0), yAxis.get("start", 0), xAxis.get("end", 1), yAxis.get("end", 1));
                mask.set("draw", (display) => {
                    display.moveTo(points[0].x, points[0].y);
                    for (let i = 1; i < points.length; i++) {
                        display.lineTo(points[i].x, points[i].y);
                    }
                    display.closePath();
                });
                xAxis.markDirtySize();
            }
        }
    }
    /**
     * @ignore
     */
    processAxis(axis) {
        this.curveContainer.children.unshift(axis);
    }
    /**
     * @ignore
     */
    inPlot(_point) {
        return true;
    }
    _tooltipToLocal(point) {
        return this.curveContainer._display.toLocal(point);
    }
    _handlePinch() {
    }
}
Object.defineProperty(CurveChart, "className", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: "CurveChart"
});
Object.defineProperty(CurveChart, "classNames", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: XYChart.classNames.concat([CurveChart.className])
});
//# sourceMappingURL=CurveChart.js.map