import { AxisRenderer } from "../xy/axes/AxisRenderer";
import { AxisLabelRadial } from "../xy/axes/AxisLabelRadial";
import { arc } from "d3-shape";
import { ListTemplate } from "../../core/util/List";
import { Template } from "../../core/util/Template";
import * as $utils from "../../core/util/Utils";
import * as $type from "../../core/util/Type";
import * as $math from "../../core/util/Math";
import * as $array from "../../core/util/Array";
/**
 * Renderer for [[CurveChart]] "vertical" axes.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/timeline/} for more info
 * @since 5.12.0
 * @important
 */
export class AxisRendererCurveY extends AxisRenderer {
    constructor() {
        super(...arguments);
        Object.defineProperty(this, "_fillGenerator", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: arc()
        });
        /**
         * A [[TemplateList]] with all the labels attached to the axis.
         *
         * `labels.template` can be used to configure appearance of the labels.
         *
         * @default new ListTemplate<AxisLabelRadial>
         */
        Object.defineProperty(this, "labels", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: this.addDisposer(new ListTemplate(Template.new({}), () => AxisLabelRadial._new(this._root, {
                themeTags: $utils.mergeTags(this.labels.template.get("themeTags", []), this.get("themeTags", []))
            }, [this.labels.template])))
        });
    }
    _afterNew() {
        this._settings.themeTags = $utils.mergeTags(this._settings.themeTags, ["renderer", "radial"]);
        super._afterNew();
        this.setPrivate("letter", "Y");
        this.setRaw("position", "absolute");
    }
    _changed() {
        super._changed();
        if (this.isDirty("axisLength")) {
            this.updateLayout();
        }
    }
    /**
     * @ignore
     */
    processAxis() {
        super.processAxis();
    }
    /**
     * @ignore
     */
    updateLayout() {
        const chart = this.chart;
        if (chart) {
            const p0 = this.positionToPoint(0, 0);
            const p1 = this.positionToPoint(1, 0);
            // draw axis
            this.set("draw", (display) => {
                display.moveTo(p0.x, p0.y);
                display.lineTo(p1.x, p1.y);
            });
            this.axis.markDirtySize();
            chart._updateMasks();
        }
    }
    /**
     * @ignore
     */
    updateGrid(grid, position, endPosition) {
        if (grid) {
            if (!$type.isNumber(position)) {
                position = 0;
            }
            let location = grid.get("location", 0.5);
            if ($type.isNumber(endPosition) && endPosition != position) {
                position = position + (endPosition - position) * location;
            }
            this.toggleVisibility(grid, position, 0, 1);
            let xRenderer = this.get("xRenderer");
            if (xRenderer) {
                const points = xRenderer.get("points");
                if (points) {
                    grid.set("draw", (display) => {
                        let previousPoint;
                        $array.each(points, (_point, index) => {
                            let pointPostion = xRenderer.pointPostion[index];
                            let p = this.positionToPoint(position, pointPostion, true);
                            if (index == 0) {
                                display.moveTo(p.x, p.y);
                            }
                            else {
                                // if distance between previous and current is very small, we skip it
                                if (previousPoint && Math.hypot(previousPoint.x - p.x, previousPoint.y - p.y) < .5) {
                                }
                                else {
                                    display.lineTo(p.x, p.y);
                                    previousPoint = p;
                                }
                            }
                        });
                    });
                }
            }
        }
    }
    // do not delete
    _handleOpposite() { }
    /**
     * Converts relative position to X/Y point.
     *
     * @param   position  Position
     * @return            Point
     */
    positionToPoint(position, positionX, doNotFix) {
        if (positionX == undefined) {
            positionX = 0;
        }
        const xRenderer = this.get("xRenderer");
        if (xRenderer) {
            return xRenderer.positionToPoint(positionX, position, doNotFix);
        }
        return { x: 0, y: 0 };
    }
    /**
     * @ignore
     */
    updateLabel(label, position, endPosition, count) {
        if (label) {
            if (!$type.isNumber(position)) {
                position = 0;
            }
            let location = 0.5;
            if ($type.isNumber(count) && count > 1) {
                location = label.get("multiLocation", location);
            }
            else {
                location = label.get("location", location);
            }
            if ($type.isNumber(endPosition) && endPosition != position) {
                position = position + (endPosition - position) * location;
            }
            const point = this.positionToPoint(position, 0);
            const xRenderer = this.get("xRenderer");
            if (xRenderer) {
                let angle = 0;
                if (xRenderer) {
                    angle = xRenderer.positionToAngle(0) - 90;
                }
                label.setAll({
                    x: point.x,
                    y: point.y
                });
                if (this.get("rotateLabels", true)) {
                    label.set("rotation", angle);
                }
            }
            this.toggleVisibility(label, position, label.get("minPosition", 0), label.get("maxPosition", 1));
        }
    }
    /**
     * @ignore
     */
    updateTick(tick, position, endPosition, count) {
        if (tick) {
            if (!$type.isNumber(position)) {
                position = 0;
            }
            let location = 0.5;
            if ($type.isNumber(count) && count > 1) {
                location = tick.get("multiLocation", location);
            }
            else {
                location = tick.get("location", location);
            }
            if ($type.isNumber(endPosition) && endPosition != position) {
                position = position + (endPosition - position) * location;
            }
            const point = this.positionToPoint(position, 0);
            const xRenderer = this.get("xRenderer");
            if (xRenderer) {
                let angle = 0;
                if (xRenderer) {
                    angle = xRenderer.positionToAngle(0) - 90;
                }
                let length = tick.get("length", 0);
                const inside = tick.get("inside");
                if (inside) {
                    length *= -1;
                }
                tick.set("draw", (display) => {
                    display.moveTo(point.x, point.y);
                    display.lineTo(point.x + length * $math.cos(angle), point.y + length * $math.sin(angle));
                });
                this.toggleVisibility(tick, position, tick.get("minPosition", 0), tick.get("maxPosition", 1));
            }
        }
    }
    /**
     * @ignore
     */
    updateBullet(bullet, position, endPosition) {
        if (bullet) {
            const sprite = bullet.get("sprite");
            if (sprite) {
                if (!$type.isNumber(position)) {
                    position = 0;
                }
                let location = bullet.get("location", 0.5);
                if ($type.isNumber(endPosition) && endPosition != position) {
                    position = position + (endPosition - position) * location;
                }
                const point = this.positionToPoint(position);
                sprite.setAll({ x: point.x, y: point.y });
                this.toggleVisibility(sprite, position, 0, 1);
            }
        }
    }
    getPoints(positionX, positionY, endPositionX, endPositionY) {
        var _a;
        return (_a = this.get("xRenderer")) === null || _a === void 0 ? void 0 : _a.getPoints(positionX, positionY, endPositionX, endPositionY);
    }
    /**
     * @ignore
     */
    updateFill(fill, position, endPosition) {
        if (fill) {
            if (position == null) {
                position = 0;
            }
            if (endPosition == null) {
                endPosition = 1;
            }
            const points = this.getPoints(0, position, 1, endPosition);
            if (points) {
                fill.set("draw", (display) => {
                    display.moveTo(points[0].x, points[0].y);
                    $array.each(points, (point) => {
                        display.lineTo(point.x, point.y);
                    });
                    display.closePath();
                });
            }
        }
    }
    /**
     * Returns axis length in pixels.
     *
     * @return Length
     */
    axisLength() {
        return this.get("axisLength", 60);
    }
    /**
     * @ignore
     */
    updateTooltipBounds(_tooltip) {
    }
    /**
     * Converts relative position to pixels.
     *
     * @param   position  Position
     * @return            Pixels
     */
    positionToCoordinate(position) {
        if (this._inversed) {
            position = Math.min(this._end, position);
            return (this._end - position) * this._axisLength;
        }
        else {
            position = Math.max(this._start, position);
            return (position - this._start) * this._axisLength;
        }
    }
    /**
     * @ignore
     */
    positionTooltip(tooltip, position) {
        const xRenderer = this.get("xRenderer");
        if (xRenderer) {
            const point = this.positionToPoint(position, xRenderer.axis.get("start", 0));
            this._positionTooltip(tooltip, point);
        }
    }
}
Object.defineProperty(AxisRendererCurveY, "className", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: "AxisRendererCurveY"
});
Object.defineProperty(AxisRendererCurveY, "classNames", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: AxisRenderer.classNames.concat([AxisRendererCurveY.className])
});
//# sourceMappingURL=AxisRendererCurveY.js.map