import { DropdownListControl } from "./DropdownListControl";
import { DrawingControl } from "./DrawingControl";
import { IndicatorControl } from "./IndicatorControl";
import { StockIcons } from "./StockIcons";
import * as $array from "../../../core/util/Array";
/**
 * A control that can be used to serialize indicators and drawings, save them
 * to local storage, and restore as needed.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/stock/toolbar/data-save-control/} for more info
 * @since 5.7.0
 */
export class DataSaveControl extends DropdownListControl {
    _afterNew() {
        super._afterNew();
        this.setPrivate("storageId", window.location.href + "-" + this.root.dom.id);
        const stockChart = this.get("stockChart");
        const dropdown = this.getPrivate("dropdown");
        // Load local storage
        if (localStorage && localStorage.getItem(this._getStorageId("autosave")) == "1") {
            this.root.events.once("frameended", () => {
                this.restoreData();
                this.set("autoSave", true);
            });
        }
        dropdown.events.on("changed", (ev) => {
            if (ev.item.id == "autosave") {
                const autoSave = !ev.item.checked;
                this.set("autoSave", autoSave);
                if (autoSave) {
                    this.saveData();
                }
            }
        });
        dropdown.events.on("invoked", (ev) => {
            if (ev.item.id == "save") {
                this.saveData();
            }
            else if (ev.item.id == "restore") {
                this.restoreData();
            }
            else if (ev.item.id == "clear") {
                this.clearData();
            }
        });
        this.on("active", () => {
            this._populateInputs();
        });
        stockChart.events.on("drawingsupdated", (_ev) => {
            if (this.get("autoSave")) {
                this.saveData();
            }
        });
        stockChart.events.on("indicatorsupdated", (_ev) => {
            if (this.get("autoSave")) {
                this.saveData();
            }
        });
    }
    _beforeChanged() {
        super._beforeChanged();
        if (this.isDirty("autoSave") && localStorage) {
            const autoSave = this.get("autoSave", false);
            if (autoSave) {
                localStorage.setItem(this._getStorageId("autosave"), "1");
                //this.saveData();
            }
            else {
                localStorage.removeItem(this._getStorageId("autosave"));
                //this.clearData();
            }
            this._populateInputs();
        }
    }
    saveData() {
        if (localStorage) {
            const drawingControl = this._getDrawingControl();
            const indicatorControl = this._getIndicatorControl();
            const drawings = drawingControl.serializeDrawings("string", "  ");
            const indicators = indicatorControl.serializeIndicators("string", "  ");
            if (drawings == "[]") {
                localStorage.removeItem(this._getStorageId("drawings"));
            }
            else {
                localStorage.setItem(this._getStorageId("drawings"), drawings);
            }
            if (indicators == "[]") {
                localStorage.removeItem(this._getStorageId("indicators"));
            }
            else {
                localStorage.setItem(this._getStorageId("indicators"), indicators);
            }
            this.events.dispatch("saved", {
                target: this,
                type: "saved",
                drawings: drawings,
                indicators: indicators
            });
        }
    }
    restoreData() {
        if (localStorage) {
            const stockChart = this.get("stockChart");
            stockChart.panels.each((panel) => {
                panel.drawings.each((drawing) => {
                    drawing.data.clear();
                });
            });
            stockChart.indicators.clear();
            const drawingControl = this._getDrawingControl();
            const indicatorControl = this._getIndicatorControl();
            const drawings = localStorage.getItem(this._getStorageId("drawings")) || "[]";
            const indicators = localStorage.getItem(this._getStorageId("indicators")) || "[]";
            drawingControl.unserializeDrawings(drawings);
            indicatorControl.unserializeIndicators(indicators);
            this.events.dispatch("restored", {
                target: this,
                type: "restored",
                drawings: drawings,
                indicators: indicators
            });
        }
    }
    clearData() {
        if (localStorage) {
            localStorage.removeItem(this._getStorageId("drawings"));
            localStorage.removeItem(this._getStorageId("indicators"));
            this.events.dispatch("cleared", {
                target: this,
                type: "cleared"
            });
        }
    }
    _getDefaultIcon() {
        return StockIcons.getIcon("Save");
    }
    _populateInputs() {
        const dropdown = this.getPrivate("dropdown");
        const items = dropdown.get("items", []);
        const autoSave = this.get("autoSave", false);
        const isSavedData = localStorage && (localStorage.getItem(this._getStorageId("drawings")) !== null || localStorage.getItem(this._getStorageId("indicators")) !== null);
        $array.each(items, (item) => {
            if (!localStorage) {
                item.disabled = true;
            }
            else if (item.id == "restore") {
                item.disabled = autoSave || !isSavedData;
            }
            else if (item.id == "clear") {
                item.disabled = !isSavedData;
            }
            else if (item.id == "save") {
                item.disabled = autoSave;
            }
            else if (item.id == "autosave") {
                item.checked = autoSave;
            }
        });
        dropdown.rebuildList();
    }
    _getStorageId(bucket) {
        return "am5-stock-" + this.get("storageId", this.getPrivate("storageId", "")) + "-" + bucket;
    }
    _getDrawingControl() {
        let drawingControl = this.getPrivate("drawingControl");
        if (drawingControl) {
            return drawingControl;
        }
        const stockChart = this.get("stockChart");
        drawingControl = stockChart.getControl("DrawingControl");
        if (!drawingControl) {
            drawingControl = DrawingControl.new(this.root, {
                stockChart: stockChart
            });
            this.setPrivate("drawingControl", drawingControl);
        }
        return drawingControl;
    }
    _getIndicatorControl() {
        let indicatorControl = this.getPrivate("indicatorControl");
        if (indicatorControl) {
            return indicatorControl;
        }
        const stockChart = this.get("stockChart");
        indicatorControl = stockChart.getControl("IndicatorControl");
        if (!indicatorControl) {
            indicatorControl = IndicatorControl.new(this.root, {
                stockChart: stockChart
            });
            this.setPrivate("indicatorControl", indicatorControl);
        }
        return indicatorControl;
    }
}
Object.defineProperty(DataSaveControl, "className", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: "DataSaveControl"
});
Object.defineProperty(DataSaveControl, "classNames", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: DropdownListControl.classNames.concat([DataSaveControl.className])
});
//# sourceMappingURL=DataSaveControl.js.map