import { ChartIndicator } from "./ChartIndicator";
import { LineSeries } from "../../xy/series/LineSeries";
/**
 * An implementation of a [[StockChart]] indicator.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/stock/indicators/} for more info
 */
export class AccumulationDistribution extends ChartIndicator {
    constructor() {
        super(...arguments);
        Object.defineProperty(this, "_editableSettings", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: [{
                    key: "seriesColor",
                    name: this.root.language.translateAny("Color"),
                    type: "color"
                }, {
                    key: "useVolume",
                    name: this.root.language.translateAny("Use Volume"),
                    type: "checkbox"
                }]
        });
    }
    _afterNew() {
        this._themeTags.push("accumulationdistribution");
        super._afterNew();
        this.yAxis.set("numberFormat", "#.###a");
    }
    _createSeries() {
        return this.panel.series.push(LineSeries.new(this._root, {
            themeTags: ["indicator"],
            xAxis: this.xAxis,
            yAxis: this.yAxis,
            valueXField: "valueX",
            valueYField: "ad",
            stroke: this.get("seriesColor"),
            fill: undefined
        }));
    }
    _prepareChildren() {
        const useVolume = "useVolume";
        if (this.isDirty(useVolume)) {
            this.markDataDirty();
            this.setCustomData(useVolume, this.get(useVolume) ? "Y" : "N");
        }
        super._prepareChildren();
    }
    /**
     * @ignore
     */
    prepareData() {
        var _a;
        if (!this.series) {
            return;
        }
        const stockSeries = this.get("stockSeries");
        const dataItems = (_a = stockSeries === null || stockSeries === void 0 ? void 0 : stockSeries.dataItems) !== null && _a !== void 0 ? _a : [];
        const volumeSeries = this.get("volumeSeries");
        this.setRaw("field", "close");
        const data = this._getDataArray(dataItems);
        const useVolume = !!this.get("useVolume");
        const volItems = useVolume && volumeSeries ? volumeSeries.dataItems : undefined;
        let prevAD = 0;
        for (let i = 0, len = dataItems.length; i < len; i++) {
            const dataItem = dataItems[i];
            const close = dataItem.get("valueY");
            if (close == null) {
                continue;
            }
            const low = dataItem.get("lowValueY", close);
            const high = dataItem.get("highValueY", close);
            const range = high - low;
            const mf = range === 0 ? 0 : ((close - low) - (high - close)) / range;
            let volume = 1;
            if (volItems) {
                const vdi = volItems[i];
                if (vdi) {
                    const v = vdi.get("valueY");
                    volume = v != null ? v : 1;
                }
            }
            const ad = prevAD + mf * volume;
            data[i].ad = ad;
            prevAD = ad;
        }
        this.series.data.setAll(data);
    }
}
Object.defineProperty(AccumulationDistribution, "className", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: "AccumulationDistribution"
});
Object.defineProperty(AccumulationDistribution, "classNames", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: ChartIndicator.classNames.concat([AccumulationDistribution.className])
});
//# sourceMappingURL=AccumulationDistribution.js.map