import { Line } from "../../../core/render/Line";
import { color } from "../../../core/util/Color";
import { DrawingSeries } from "./DrawingSeries";
import * as $array from "../../../core/util/Array";
/**
 * Draws a multi-point line (polyline).
 */
export class PolylineSeries extends DrawingSeries {
    constructor() {
        super(...arguments);
        Object.defineProperty(this, "_tag", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: "polyline"
        });
        Object.defineProperty(this, "_drawingLine", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: this.mainContainer.children.push(Line.new(this._root, { forceInactive: true }))
        });
    }
    _prepareChildren() {
        super._prepareChildren();
        this.strokes.template.setAll({
            fill: color(0xffffff),
            fillOpacity: 0
        });
        if (this.isDirty("fillShape")) {
            if (this.get("fillShape")) {
                this.fills.template.setAll({
                    visible: true,
                    forceHidden: false,
                    fillOpacity: this.get("fillOpacity"),
                    fill: this.get("fillColor")
                });
            }
            else {
                this.fills.template.setAll({
                    visible: false
                });
            }
        }
    }
    _handlePointerClick(event) {
        if (this._drawingEnabled) {
            super._handlePointerClick(event);
            if (event.target.get("userData") == "grip") {
                this._endPolyline(event.target.dataItem);
            }
            else {
                if (!this._isDragging) {
                    this.isDrawing(true);
                    this._hideResizer(event.target);
                    if (this.unselectAllDrawings() == 0) {
                        // for consistency with other series
                        if (this._index == 0) {
                            this._index = 1;
                        }
                        if (this._pIndex == 0) {
                            this._increaseIndex();
                            const context = { stroke: this._getStrokeTemplate(), fill: this._getFillTemplate(), index: this._index, corner: "e", drawingId: this._drawingId };
                            this._setContextSprite(context);
                            this.data.push(context);
                        }
                        this._drawingLine.show();
                        this._addPoint(event);
                        // add one more if fill
                        if (this.get("fillShape")) {
                            if (this._pIndex == 1) {
                                this._addPoint(event, true);
                            }
                            else if (this._pIndex > 1) {
                                this.data.moveValue(this.data.getIndex(this.data.length - 1), this.data.length - 2);
                            }
                        }
                        if (this._pIndex - 1 == this.get("pointCount", 1000)) {
                            this._endPolyline();
                            return;
                        }
                    }
                }
                this._drawingLine.set("stroke", this.get("strokeColor"));
            }
        }
    }
    _setContextSprite(context) {
        context.sprite = this.mainContainer;
    }
    disableDrawing() {
        super.disableDrawing();
        if (this._pIndex > 0) {
            this._endPolyline();
        }
    }
    clearDrawings() {
        super.clearDrawings();
        this._drawingLine.hide();
    }
    _addPoint(event, closing) {
        const chart = this.chart;
        if (chart) {
            const xAxis = this.get("xAxis");
            const yAxis = this.get("yAxis");
            const point = chart.plotContainer.toLocal(event.point);
            const valueX = this._getXValue(xAxis.positionToValue(xAxis.coordinateToPosition(point.x)));
            const valueY = this._getYValue(yAxis.positionToValue(yAxis.coordinateToPosition(point.y)), valueX);
            const dataItems = this.dataItems;
            const len = dataItems.length;
            this.data.push({ valueY: valueY, valueX: valueX, index: this._index, corner: this._pIndex, drawingId: this._drawingId, closing: closing });
            this.setPrivate("startIndex", 0);
            this.setPrivate("endIndex", len);
            const dataItem = dataItems[len];
            this._positionBullets(dataItem);
            this._setXLocation(dataItem, valueX);
            this._pIndex++;
        }
    }
    _endPolyline(dataItem) {
        if (!dataItem) {
            dataItem = this.dataItems[this.dataItems.length - 1];
        }
        if (dataItem) {
            this._pIndex = 0;
            const dataContext = dataItem.dataContext;
            const index = dataContext.index;
            if (dataContext.corner == 0) {
                this.data.push({ valueX: dataItem.get("valueX"), valueY: dataItem.get("valueY"), index: index, corner: "c", closing: true, drawingId: this._drawingId });
                const dataItems = this.dataItems;
                const len = dataItems.length - 1;
                this.setPrivate("startIndex", 0);
                this.setPrivate("endIndex", len);
                dataItem = dataItems[len];
                this._positionBullets(dataItem);
                this._setXLocation(dataItem, dataItem.get("valueX", 0));
            }
            this._drawingLine.hide();
            this.isDrawing(false);
            this._dispatchAdded();
        }
    }
    _dispatchAdded() {
        this._dispatchStockEvent("drawingadded", this._drawingId, this._index);
    }
    _handlePointerMove(event) {
        super._handlePointerMove(event);
        if (this._isDrawing) {
            const movePoint = this._movePointerPoint;
            if (movePoint) {
                const dataItems = this.dataItems;
                const len = dataItems.length;
                if (len > 0) {
                    const lastItem = dataItems[len - 1];
                    const points = [];
                    const point = lastItem.get("point");
                    if (point) {
                        points.push(point);
                    }
                    points.push(movePoint);
                    if (this.get("fillShape")) {
                        const bLastItem = dataItems[len - 2];
                        if (bLastItem) {
                            const bPoint = bLastItem.get("point");
                            if (bPoint) {
                                points.push(bPoint);
                            }
                        }
                    }
                    if (points.length > 1) {
                        this._drawingLine.set("points", points);
                    }
                }
            }
        }
    }
    _updateElements() {
        $array.each(this.dataItems, (dataItem) => {
            const dataContext = dataItem.dataContext;
            if (dataContext) {
                const closing = dataContext.closing;
                if (closing) {
                    if (this._di[dataContext.index]) {
                        const closingDataItem = this._di[dataContext.index][0];
                        const valueX = closingDataItem.get("valueX", 0);
                        const valueY = closingDataItem.get("valueY");
                        this._setContext(dataItem, "valueX", valueX);
                        this._setContext(dataItem, "valueY", valueY, true);
                        this._setXLocation(dataItem, valueX);
                        this._positionBullets(dataItem);
                        const bullets = dataItem.bullets;
                        if (bullets) {
                            $array.each(bullets, (bullet) => {
                                const sprite = bullet.get("sprite");
                                if (sprite) {
                                    sprite.set("forceHidden", true);
                                }
                            });
                        }
                    }
                }
            }
        });
    }
    /**
     * Cancels current drawing
     *
     * @since 5.9.0
     */
    cancelDrawing() {
        super.cancelDrawing();
        this._drawingLine.hide(0);
    }
}
Object.defineProperty(PolylineSeries, "className", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: "PolylineSeries"
});
Object.defineProperty(PolylineSeries, "classNames", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: DrawingSeries.classNames.concat([PolylineSeries.className])
});
//# sourceMappingURL=PolylineSeries.js.map