import type { Gantt } from "./Gantt";
import type { GanttCategoryAxis, IGanttCategoryAxisDataItem } from "./GanttCategoryAxis";
import type { GanttCategoryAxisRenderer } from "./GanttCategoryAxisRenderer";
import type { GanttDateAxis } from "./GanttDateAxis";
import type { GanttDateAxisRenderer } from "./GanttDateAxisRenderer";
import { ColumnSeries, IColumnSeriesPrivate, IColumnSeriesSettings, IColumnSeriesDataItem, IColumnSeriesAxisRange } from "../xy/series/ColumnSeries";
import { Container } from "../../core/render/Container";
import { Circle, DataItem, Graphics, IPoint, ISpritePointerEvent, Line, Link, ListTemplate, Rectangle, RoundedRectangle, Triangle } from "../../..";
export interface IGanttSeriesDataItem extends IColumnSeriesDataItem {
    /**
     * [[Container]] that holds all the elements of the series item, except the column graphics itslef.
     */
    container?: Container;
    /**
     * A [[Container]] that is masked and holds progress rectangle and progress grip.
     */
    maskedContainer?: Container;
    /**
     * A [[RoundedRectangle]] that is used to mask the progress rectangle.
     */
    mask?: RoundedRectangle;
    /**
     * A [[Circle]] used as a bullet shown at the start of the task bar.
     */
    startBullet?: Circle;
    /**
     * A [[Circle]] used as a bullet shown at the end of the task bar.
     */
    endBullet?: Circle;
    /**
     * A [[Rectangle]] filled with line pattern which is used to resize task bar.
     */
    startGrip?: Rectangle;
    /**
     * A [[Rectangle]] filled with line pattern which is used to resize task bar.
     */
    endGrip?: Rectangle;
    /**
     * A [[Rectangle]] that is used to show progress of the task. It actually
     * shows the remaining part of the task and is filled with diagonal line
     * pattern.
     */
    progressRectangle?: Rectangle;
    /**
     * A [[RoundedRectangle]] rotated by 45 degrees and is shown instead of a
     * column when the task has zero duration.
     */
    zeroRectangle?: RoundedRectangle;
    /**
     * A [[Triangle]] that is used to resize progress rectangle.
     */
    progressGrip?: Triangle;
    /**
     * A value that holds the progress value of the task.
     */
    progress?: number;
    /**
     * A value that holds previous progress value of the task, used when
     * toggling progress using progress pie next to the y axis label.
     */
    prevProgress?: number;
    /**
     * Duration of the task, in units of the x axis.
     */
    duration?: number;
    /**
     * Array of IDs of tasks this task is linked to.
     */
    linkTo?: Array<string>;
    /**
     * Refferences to `Link` objects that are used to visually connect tasks.
     */
    links?: {
        [index: string]: Link;
    };
    /**
     * A reference to a corresponding category axis data item.
     */
    categoryAxisDataItem?: DataItem<IGanttCategoryAxisDataItem>;
    /**
     * Array of children data items.
     */
    children?: Array<DataItem<IGanttSeriesDataItem>>;
    /**
     * A reference to a parent data item, if any.
     */
    parent?: DataItem<IGanttSeriesDataItem>;
    /**
     * Category name.
     */
    name?: string;
}
export interface IGanttSeriesSettings extends IColumnSeriesSettings {
    /**
     * A field in data that holds progress of the task.
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Series_data} for more info
     * @default "progress"
     */
    progressField?: string;
    /**
     * A field in data that holds duration of the task.
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Series_data} for more info
     * @default "duration"
     */
    durationField?: string;
    /**
     * A field in data that holds and ID of the task it is linked to.
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Series_data} for more info
     * @default "linkTo"
     */
    linkToField?: string;
    /**
     * A distance in pixels that link should be protracted from the edge of the
     * task bars.
     *
     * @default 25
     */
    linkHorizontalOffset?: number;
    /**
     * When dragging/resizing a column, how many units should it snap to.
     * @default 1
     */
    snapCount?: number;
    /**
     * A reference to the y-axis of the Gantt chart.
     */
    yAxis: GanttCategoryAxis<GanttCategoryAxisRenderer>;
    /**
     * A reference to the x-axis of the Gantt chart.
     */
    xAxis: GanttDateAxis<GanttDateAxisRenderer>;
}
export interface IGanttSeriesPrivate extends IColumnSeriesPrivate {
}
export interface IGanttSeriesAxisRange extends IColumnSeriesAxisRange {
}
/**
 * A series used in [[Gantt]] chart to display tasks and their progress.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Gantt_series} for more info
 * @important
 * @since 5.14.0
 */
export declare class GanttSeries extends ColumnSeries {
    _settings: IGanttSeriesSettings;
    _privateSettings: IGanttSeriesPrivate;
    _dataItemSettings: IGanttSeriesDataItem;
    _axisRangeType: IGanttSeriesAxisRange;
    static className: string;
    static classNames: Array<string>;
    protected _startBullet?: Circle;
    protected _endBullet?: Circle;
    /**
     * A reference to the parent [[Gantt]] chart.
     */
    gantt: Gantt;
    protected _columnDragStartX?: number;
    protected _columnDragStartY?: number;
    protected _hoveredDataItem?: DataItem<IGanttSeriesDataItem>;
    protected _xPan?: boolean;
    protected _yPan?: boolean;
    /**
     * A container that holds all the links between tasks.
     */
    linksContainer: Container;
    /**
     * A line which is shown while creating a connector between two tasks.
     */
    connectorLine: Line;
    /**
     * A triangle that is shown at the end of the connector line, while creating
     * a connector.
     */
    connectorArrow: Triangle;
    /**
     * [[ListTemplate]] of [[Link]] that connect tasks.
     */
    readonly links: ListTemplate<Link>;
    /**
     * [[ListTemplate]] of [[Container]]s that hold all the elements of series
     * items, such as grips, bullets, etc.
     */
    readonly containers: ListTemplate<Container>;
    /**
     * [[ListTemplate]] of [[Container]]s that are used to mask elements, such
     * as progress rectangles.
     */
    readonly maskedContainers: ListTemplate<Container>;
    /**
     * [[ListTemplate]] of [[Circle]]s that are used as start bullets.
     */
    readonly startBullets: ListTemplate<Circle>;
    /**
     * [[ListTemplate]] of [[Circle]]s that are used as end bullets.
     */
    readonly endBullets: ListTemplate<Circle>;
    /**
     * [[ListTemplate]] of [[Rectangle]]s that are used to resize task bars.
     */
    readonly startGrips: ListTemplate<Rectangle>;
    /**
     * [[ListTemplate]] of [[Rectangle]]s that are used to resize task bars.
     */
    readonly endGrips: ListTemplate<Rectangle>;
    /**
     * [[ListTemplate]] of [[RoundedRectangle]]s that are used to show
     * zero-duration tasks.
     */
    readonly zeroRectangles: ListTemplate<RoundedRectangle>;
    /**
     * [[ListTemplate]] of [[Rectangle]]s that are used to show progress of the
     * task. It is actually a remaining part of the task and is filled with
     * diagonal line pattern.
     */
    readonly progressRectangles: ListTemplate<Rectangle>;
    /**
     * [[ListTemplate]] of [[Triangle]]s that are used to resize progress
     * rectangles.
     */
    readonly progressGrips: ListTemplate<Triangle>;
    protected _afterNew(): void;
    /**
     * @ignore
     */
    protected roundValue(value: number, invertThreshold?: boolean): number;
    /**
     * @ignore
     */
    makeColumn(dataItem: DataItem<this["_dataItemSettings"]>, listTemplate: ListTemplate<RoundedRectangle>): RoundedRectangle;
    protected _handleColumnDragged(e: ISpritePointerEvent): void;
    protected _handleColumnHover(e: ISpritePointerEvent): void;
    /**
     * Checks if the task is collapsed.
     *
     * @param  dataItem  Data item
     * @return           Collapsed?
     */
    isCollapsed(dataItem: DataItem<IGanttSeriesDataItem>): boolean;
    /**
     * Updates the series, recalculating values and updating children.
     *
     * @ignore Exclude from docs
     */
    _updateChildren(): void;
    protected _getParentOpen(dataItem: DataItem<IGanttSeriesDataItem>): number;
    protected _adjustLinkValues(dataItem: DataItem<IGanttSeriesDataItem>): void;
    protected _adjustChildValues(dataItem: DataItem<IGanttSeriesDataItem>): void;
    protected _dragChildValues(dataItem: DataItem<IGanttSeriesDataItem>, dx: number): void;
    protected _moveChildValues(dataItem: DataItem<IGanttSeriesDataItem>, diff: number): void;
    protected _findMinStart(dataItem: DataItem<IGanttSeriesDataItem>): number;
    protected _adjustParentValues(dataItem: DataItem<IGanttSeriesDataItem>): void;
    protected _startConnector(bullet: Circle): void;
    protected _updateConnector(point: IPoint): void;
    protected _endConnector(): void;
    /**
     * Fixes values of a data item, adjusting start, end, or duration as needed.
     */
    protected _fixValues(dataItem: DataItem<IGanttSeriesDataItem>, keep: "start" | "end" | "duration"): void;
    protected _toggleColumn(dataItem: DataItem<this["_dataItemSettings"]>, visible: boolean): void;
    protected _afterDataChange(): void;
    protected _updateLinks(): void;
    /**
     * Updates series graphics.
     */
    protected _updateSeriesGraphics(dataItem: DataItem<this["_dataItemSettings"]>, graphics: Graphics, l: number, r: number, t: number, b: number, _fitW: boolean, fitH: boolean): void;
    /**
     * Hides series's data item.
     *
     * @param   dataItem  Data item
     * @param   duration  Animation duration in milliseconds
     * @return            Promise
     */
    hideDataItem(dataItem: DataItem<this["_dataItemSettings"]>, duration?: number): Promise<void>;
    /**
     * Shows series's data item.
     *
     * @param   dataItem  Data item
     * @param   duration  Animation duration in milliseconds
     * @return            Promise
     */
    showDataItem(dataItem: DataItem<this["_dataItemSettings"]>, duration?: number): Promise<void>;
    /**
     * Disposes a data item and removes it from the series.
     *
     * @param dataItem  Data item
     */
    disposeDataItem(dataItem: DataItem<this["_dataItemSettings"]>): void;
    /**
     * Disposes all links of a data item and removes them from the series.
     *
     * @param dataItem  Data item
     */
    disposeLinks(dataItem: DataItem<IGanttSeriesDataItem>): void;
    /**
     * Returns the duration of a data item in units.
     *
     * @param dataItem  Data item
     * @return Duration in units
     */
    getDataItemDuration(dataItem: DataItem<IGanttSeriesDataItem>): number;
    /**
     * Returns the depth of a data item in the hierarchy.
     *
     * @param dataItem  Data item
     * @return Depth level
     */
    getDepth(dataItem: DataItem<IGanttSeriesDataItem>): number;
    /**
     * Returns the end value of a data item, adjusting for holidays and weekends.
     *
     * @param openValue  Open value in milliseconds
     * @param duration   Duration in units
     * @return           Adjusted end value in milliseconds
     */
    getEndValue(openValue: number, duration: number): number;
    /**
     * Returns the open value of a data item, adjusting for holidays and weekends.
     *
     * @param openValue  Open value in milliseconds
     * @return           Adjusted open value
     */
    getOpenValue(openValue: number): number;
    /**
     * Returns the duration of a unit in milliseconds.
     *
     * @return Duration in milliseconds
     */
    getUnitDuration(): number;
    /**
     * Returns the count of holidays between two dates.
     *
     * @param start  Start date in milliseconds
     * @param end    End date in milliseconds
     * @return       Count of holidays
     */
    getHolidayCount(start: number, end: number): number;
    /**
         * Checks if a given date is a holiday or weekend.
         *
         * @param value  Date value in milliseconds
         * @return       `true` if the date is a holiday or weekend, otherwise `false`
         */
    isHoliday(value: number): boolean;
}
//# sourceMappingURL=GanttSeries.d.ts.map