import type { AxisRenderer } from "../xy/axes/AxisRenderer";
import type { GanttCategoryAxisRenderer } from "./GanttCategoryAxisRenderer";
import type { Container } from "../../core/render/Container";
import type { ProgressPie } from "../../core/render/ProgressPie";
import type { Gantt } from "./Gantt";
import type { AxisTick } from "../xy/axes/AxisTick";
import type { NumericStepper } from "../../core/render/NumericStepper";
import type { IGanttSeriesDataItem } from "./GanttSeries";
import type { Animation } from "../../core/util/Entity";
import { Rectangle } from "../../core/render/Rectangle";
import { CategoryAxis, ICategoryAxisSettings, ICategoryAxisPrivate, ICategoryAxisDataItem, ICategoryAxisEvents } from "../xy/axes/CategoryAxis";
import { Button, Color, DataItem, IDisposer } from "../../..";
export interface IGanttCategoryAxisSettings<R extends AxisRenderer> extends ICategoryAxisSettings<R> {
    /**
     * Currently selected category data item, if any.
     */
    selectedDataItem?: DataItem<IGanttCategoryAxisDataItem>;
    /**
     * A field in data that holds parent id.
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Category_data} for more info
     * @default "parentId"
     */
    parentIdField?: string;
    /**
     * A field in data that holds status whether the category is collapsed.
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Category_data} for more info
     * @default "collapsed"
     */
    collapsedField?: string;
    /**
     * Size of child categories relative to the top-level cell height. (`0` - `1`)
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Cell_height_for_child_categories} for more info
     * @default 0.8
     */
    childCellSize?: number;
    /**
     * A shift in pixels to apply to child categories.
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Cell_height_for_child_categories} for more info
     * @default 25
     */
    childShift?: number;
    /**
     * A minimal height of the cell in pixels.
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Cell_height} for more info
     * @default 70
     */
    minCellHeight?: number;
    /**
     * Field of a category name
     * @default "name"
     */
    nameField?: string;
    /**
     * Field of a category color
     * @default "color"
     */
    colorField?: string;
}
export interface IGanttCategoryAxisDataItem extends ICategoryAxisDataItem {
    /**
     * A container that holds all other elements of a axis label - label, controls, grip, etc.
     */
    container?: Container;
    /**
     * A container that holds progress pie and duration stepper.
     */
    controlsContainer?: Container;
    /**
     * A grip for dragging the category.
     */
    grip?: Rectangle;
    /**
     * A bullet to the left of a label (circle or triangle, if it has children)
     */
    taskBullet?: Button;
    /**
     * A progress pie that shows progress of the task.
     */
    progressPie?: ProgressPie;
    /**
     * Children of this category, if any.
     */
    children?: Array<DataItem<IGanttCategoryAxisDataItem>>;
    /**
     * Parent id of data item.
     */
    parentId?: string;
    /**
     * A reference to the parent category data item.
     */
    parent?: DataItem<IGanttCategoryAxisDataItem>;
    /**
     * Progress of the task, from `0` to `1`. If this item has children, this will be the average of all children's progress.
     */
    progress?: number;
    /**
     * A stepper that allows to change task duration.
     */
    durationStepper?: NumericStepper;
    /**
     * A flag indicating whether the category is collapsed.
     */
    collapsed?: boolean;
    /**
     * Duration of the task (in days or other units, depending on `durationUnit` setting of a `Gantt`).
     */
    duration?: number;
    /**
     * Color of a task.
     */
    color?: Color;
    /**
     * Custom color for the task, if any.
     */
    customColor?: Color;
    /**
     * Displayed name of a category.
     */
    name?: string;
    /**
     * @ignore
     */
    deleting?: boolean;
    /**
     * A reference to the series data item.
     */
    seriesDataItem?: DataItem<IGanttSeriesDataItem>;
}
export interface IGanttCategoryAxisPrivate extends ICategoryAxisPrivate {
}
export interface IGanttCategoryAxisEvents extends ICategoryAxisEvents {
}
/**
 * A category axis that is used as a Y (vertical) axis for [[Gantt]] charts.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Category_vertical_axis} for more info
 * @since 5.14.0
 * @important
 */
export declare class GanttCategoryAxis<R extends GanttCategoryAxisRenderer> extends CategoryAxis<R> {
    static className: string;
    static classNames: Array<string>;
    _settings: IGanttCategoryAxisSettings<R>;
    _privateSettings: IGanttCategoryAxisPrivate;
    _dataItemSettings: IGanttCategoryAxisDataItem;
    _events: IGanttCategoryAxisEvents;
    protected _downX: number | undefined;
    protected _downW: number | undefined;
    protected _xHideDP: IDisposer | undefined;
    protected _insertAfter: DataItem<IGanttCategoryAxisDataItem> | undefined;
    protected _makeChildOf: DataItem<IGanttCategoryAxisDataItem> | undefined;
    protected _draggedItem: DataItem<IGanttCategoryAxisDataItem> | undefined;
    protected _previousTick: AxisTick | undefined;
    protected _delDp: IDisposer | undefined;
    protected _firstDataItem: DataItem<IGanttCategoryAxisDataItem>;
    protected _updDP: IDisposer | undefined;
    protected _dataChangeDp: IDisposer | undefined;
    protected _treeDirty: boolean;
    /**
     * A reference to the parent [[Gantt]] chart.
     */
    gantt?: Gantt;
    /**
     * A [[Button]] used for deleting categories.
     */
    xButton: Button;
    protected _afterNew(): void;
    _changed(): void;
    protected _processBullet(dataItem: DataItem<IGanttCategoryAxisDataItem>): void;
    _updateChildren(): void;
    _clearDirty(): void;
    getSeriesDataItem(dataItem: DataItem<IGanttCategoryAxisDataItem>): DataItem<IGanttSeriesDataItem> | undefined;
    /**
     * Zooms the axis to relative locations.
     *
     * Both `start` and `end` are relative: 0 means start of the axis, 1 - end.
     *
     * @param   start     Relative start
     * @param   end       Relative end
     * @param   duration  Duration of the zoom animation in milliseconds
     * @return            Zoom animation
     */
    zoom(start: number, end: number, duration?: number): Animation<this["_settings"]["start"]> | Animation<this["_settings"]["end"]> | undefined;
    _afterDataChange(): void;
    markDirtyTree(): void;
    _sortCategoryAxis(): void;
    _setNewColor(dataItem: DataItem<IGanttCategoryAxisDataItem>, color?: Color): void;
    _updateTree(): void;
    _updateCellSizes(duration?: number): void;
    _updateBullet(dataItem: DataItem<IGanttCategoryAxisDataItem>): void;
    /**
     * Toggles collapse state of a data item.
     * @param dataItem
     * @param collapse
     * @param duration
     */
    toggleCollapse(dataItem: DataItem<IGanttCategoryAxisDataItem>, collapse?: boolean, duration?: number): void;
    /**
     * Expands all categories in the axis.
     */
    expandAll(): void;
    /**
     * Collapses all categories in the axis.
     */
    collapseAll(): void;
    /**
     * Deletes all data items from the axis.
     */
    deleteAll(): void;
    /**
     * Deletes a data item from the axis.
     *
     * @param dataItem  Data item to delete
     */
    deleteDataItem(dataItem: DataItem<IGanttCategoryAxisDataItem>, child?: boolean): void;
    /**
     * Toggles progress pie for a data item. Toggling means that progress pie
     * will animate to 1 if it was less than 1, or animate to previous progress
     * value if it was 1.
     *
     * @param dataItem  Data item to toggle progress pie for
     */
    toggleProgressPie(dataItem?: DataItem<IGanttCategoryAxisDataItem>, value?: number): void;
    /**
     * Sets a custom color for a data item.
     *
     * @param dataItem  Data item to set color for
     * @param c         Color to set
     */
    setDataItemColor(dataItem?: DataItem<IGanttCategoryAxisDataItem>, c?: Color): void;
    /**
     * Hides axis's data item.
     *
     * @param   dataItem  Data item
     * @param   duration  Animation duration in milliseconds
     * @return            Promise
     */
    hideDataItem(dataItem: DataItem<IGanttCategoryAxisDataItem>): Promise<void>;
    /**
     * Shows axis's data item.
     *
     * @param   dataItem  Data item
     * @param   duration  Animation duration in milliseconds
     * @return            Promise
     */
    showDataItem(dataItem: DataItem<IGanttCategoryAxisDataItem>): Promise<void>;
    _toggleFHDataItem(dataItem: DataItem<IGanttCategoryAxisDataItem>, forceHidden: boolean): void;
    _toggleDataItem(dataItem: DataItem<IGanttCategoryAxisDataItem>, visible: boolean): void;
    /**
     * Disposes a data item.
     *
     * @param dataItem  Data item to dispose
     */
    disposeDataItem(dataItem: DataItem<this["_dataItemSettings"]>): void;
    /**
     * Selects a data item and shows the delete button. If the data item is
     * already selected, it will unselect it.
     *
     * @param dataItem  Data item to select
     */
    selectDataItem(dataItem?: DataItem<IGanttCategoryAxisDataItem>): void;
    /**
     * Unselects all data items.
     */
    unselectDataItems(): void;
    /**
     * @ignore
     */
    _disposeXHideDP(): void;
    /**
     * @ignore
     * @param dataItem
     * @returns depth of the data item in the hierarchy
     */
    _getItemDepth(dataItem: DataItem<IGanttCategoryAxisDataItem>): number;
    /**
     * @ignore
     * Overrides original method so that a ghost label is not created.
     */
    protected _createGhostLabel(): void;
}
//# sourceMappingURL=GanttCategoryAxis.d.ts.map