import type { ITimeInterval } from "../../core/util/Time";
import { Button } from "../../core/render/Button";
import { ConfirmButton } from "../../core/render/ConfirmButton";
import { Percent } from "../../core/util/Percent";
import { GanttDateAxis, IGanttDateAxisDataItem } from "./GanttDateAxis";
import { XYChart } from "../xy/XYChart";
import { GanttSeries } from "./GanttSeries";
import { Color, ColorSet, Container, DataItem, IContainerEvents, IContainerPrivate, IContainerSettings, Scrollbar } from "../../..";
import { GanttCategoryAxis } from "./GanttCategoryAxis";
import { GanttCategoryAxisRenderer } from "./GanttCategoryAxisRenderer";
import { GanttDateAxisRenderer } from "./GanttDateAxisRenderer";
import { ColorPicker } from "../../plugins/colorPicker/ColorPicker";
import { ColorPickerButton } from "../../plugins/colorPicker/ColorPickerButton";
export interface IGanttSettings extends IContainerSettings {
    /**
     * If set to `true`, the user will be able to edit the chart via UI.
     *
     * @default true
     */
    editable?: boolean;
    /**
     * A unit to be used for when calculating "duration" of Gantt items.
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Duration_units} for more information
     * @default "day"
     */
    durationUnit?: "year" | "month" | "week" | "day" | "hour" | "minute" | "second";
    /**
     * An array of weekend days, e.g. `[0, 6]` for Sunday and Saturday.
     *
     * @default [0, 6]
     * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Weekends_and_holidays} for more information
     */
    weekends?: Array<number>;
    /**
     * An array of dates to treat as a holiday.
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Weekends_and_holidays} for more information
     */
    holidays?: Array<Date>;
    /**
     * If set to `true`, weekends will be excluded from the chart.
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Weekends_and_holidays} for more information
     */
    excludeWeekends?: boolean;
    /**
     * An absolute or relative width of the left-side category column.
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Category_axis_width} for more information
     * @default 30%
     */
    sidebarWidth?: number | Percent;
    /**
     * A relative vertical cell position to treat as the threshold for snapping
     * bars.
     *
     * Available range of values: `0` (left) to `1` (right).
     *
     * This will be used when resizing or dragging a bar. If the position is
     * bigger than `snapThreshold` the bar will snap to the right, if it is
     * smaller than `snapThreshold`, it will snap to the left.
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Snapping_behavior} for more information
     * @default 0.5
     */
    snapThreshold?: number;
    /**
     * A [[ColorSet]] to use when asigning colors for series.
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/series/#Series_colors} for more info
     */
    colors?: ColorSet;
    /**
     * Grid intervals.
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/#Timeline_horizontal_axis} for more info
     */
    gridIntervals?: {
        [index: string]: Array<ITimeInterval>;
    };
    /**
     * If this is set to `true`, when a new task is added, it will be automatically linked to the previous task.
     *
     * @default true
     */
    linkNewTasks?: boolean;
}
export interface IGanttPrivate extends IContainerPrivate {
}
export interface IGanttEvents extends IContainerEvents {
    datemarked: {
        date: number | undefined;
        dataItem: DataItem<IGanttDateAxisDataItem>;
    };
    dateunmarked: {
        date: number | undefined;
        dataItem?: DataItem<IGanttDateAxisDataItem>;
    };
    valueschanged: {};
}
/**
 * Creates a [[Gantt]] chart.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/gantt/} for more info
 * @important
 * @since 5.14.0
 */
export declare class Gantt extends Container {
    _settings: IGanttSettings;
    _privateSettings: IGanttPrivate;
    _events: IGanttEvents;
    static className: string;
    static classNames: Array<string>;
    /**
     * A scrollbar for horizontal scrolling.
     */
    scrollbarX: Scrollbar;
    /**
     * A scrollbar for vertical scrolling.
     */
    scrollbarY: Scrollbar;
    /**
     * A reference to the [[XYChart]] that holds the Gantt chart.
     */
    xyChart: XYChart;
    /**
     * A reference to the [[GanttSeries]] that shows the Gantt tasks bars.
     */
    series: GanttSeries;
    /**
     * A reference to the [[GanttDateAxis]] used as the main X-axis.
     */
    xAxis: GanttDateAxis<GanttDateAxisRenderer>;
    /**
     * A reference to the [[GanttDateAxis]] used as the secondary X-axis.
     */
    xAxisMinor: GanttDateAxis<GanttDateAxisRenderer>;
    /**
     * A reference to the [[GanttCategoryAxis]] used as the Y-axis.
     */
    yAxis: GanttCategoryAxis<GanttCategoryAxisRenderer>;
    /**
     * Controls (add, color picker, expand, collaps, clear buttons) container.
     */
    controls: Container;
    zoomControls: Container;
    /**
     * The [[Button]] element to add new tasks.
     */
    readonly addButton: Button;
    /**
     * The [[ColorPickerButton]] element to select colors for tasks.
     */
    readonly colorPickerButton: ColorPickerButton;
    /**
     * The [[Button]] elements to expand and collapse all tasks.
     */
    readonly expandButton: Button;
    /**
     * The [[Button]] elements to collapse all tasks.
     */
    readonly collapseButton: Button;
    /**
     * The [[Button]] element to toggle `linkNewTasks` setting.
     */
    readonly linkButton: Button;
    /**
     * The [[Button]] element to horizontally fit visible tasks into a view.
     */
    readonly fitButton: Button;
    /**
     * The [[Button]] element to zoom out the X axis.
     */
    readonly zoomOutButton: Button;
    /**
     * The [[Button]] element to toggle edit mode.
     *
     * @since 5.14.1
     */
    readonly editButton: Button;
    /**
     * The [[Button]] element to clear all tasks.
     */
    readonly clearButton: ConfirmButton;
    /**
     * The [[ColorPicker]] element to select colors for tasks.
     */
    readonly colorPicker: ColorPicker;
    protected _customColor: Color | undefined;
    protected _afterNew(): void;
    _prepareChildren(): void;
    clearAll(): void;
    _updateChildren(): void;
    protected _updateScrollbar(): void;
    /**
     * Adds a new task to the Gantt chart.
     *
     * @param category        The category name for the task. If not provided, will be auto-generated.
     * @param start           The start time of the task in milliseconds. If not provided, will be determined based on context.
     * @param duration        The duration of the task. If not provided, defaults to 1.
     * @param parentId        The parent category if this is a subtask.
     * @param progress        The initial progress of the task (0-1). Defaults to 0.
     * @param linkToPrevious  Should the task automatically link to the previous one?
     */
    addNewTask(name?: string, start?: number, duration?: number, parentId?: string, progress?: number, linkToPrevious?: boolean): void;
    _nextColor(): Color | undefined;
    /**
     * Marks a date on the minor date axis.
     *
     * @param date  Date to be marked
     */
    markDate(date: number): void;
    /**
     * Unmarks a date on the minor date axis.
     *
     * @param date  Date to be unmarked
     */
    unmarkDate(date: number): void;
    protected _toggleEditable(value: boolean): void;
}
//# sourceMappingURL=Gantt.d.ts.map