<?php

namespace App\Http\Controllers;

use Inertia\Inertia;
use Illuminate\Http\Request;
use App\Models\Access\Role;
use App\Base\Constants\Auth\Role as RoleSlug;

class RoleController extends Controller
{
    /**
     * Show roles page
     */
    public function index(Request $request)
    {
        $adminCreatedRoles = Role::whereNull('created_by')->get();
        $superAdmin = auth()->user();

        foreach ($adminCreatedRoles as $role) {
            $role->update(['created_by' => $superAdmin->id]);
        }

        return Inertia::render('pages/roles/index');
    }

    /**
     * Get roles list (AJAX)
     */
    public function getRoles(Request $request)
    {
        $perPage = $request->input('perPage', 10);
        $searchQuery = $request->input('searchQuery');
        $user = auth()->user();

        $exceptRoles = RoleSlug::mobileAppRoles();
        $exceptRoles[] = RoleSlug::DISPATCHER;

        $roleQuery = Role::where('created_by', $user->id)
            ->where('slug', '!=', 'super-admin')
            ->whereNotIn('slug', $exceptRoles);

        if ($searchQuery) {
            $roleQuery->where(function ($query) use ($searchQuery) {
                $query->where('slug', 'like', "%{$searchQuery}%")
                      ->orWhere('name', 'like', "%{$searchQuery}%")
                      ->orWhere('description', 'like', "%{$searchQuery}%");
            });
        }

        $roles = $roleQuery->latest()->paginate($perPage);

        return response()->json([
            'roles' => $roles
        ]);
    }

    /**
     * Store new role
     */
    public function store(Request $request)
    {
        if (env('APP_FOR') === 'demo') {
            return response()->json([
                'alertMessage' => 'You are not Authorized'
            ], 403);
        }

        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        $currentUser = auth()->user();

        $validatedData['slug'] = $this->convertToSlug($validatedData['name']);
        $validatedData['created_by'] = $currentUser->id;

        $role = Role::create($validatedData);

        return response()->json([
            'role' => $role,
            'successMessage' => 'Role created successfully!'
        ], 201);
    }

    /**
     * Update role
     */
    public function update(Request $request, Role $role)
    {
        if (env('APP_FOR') === 'demo') {
            return response()->json([
                'alertMessage' => 'You are not Authorized'
            ], 403);
        }

        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        $validatedData['slug'] = $this->convertToSlug($validatedData['name']);

        $role->update($validatedData);

        return response()->json([
            'role' => $role,
            'successMessage' => 'Role updated successfully'
        ]);
    }

    /**
     * Delete role  ✅ FIXED
     */
    public function destroy(Role $role)
    {
        if (env('APP_FOR') === 'demo') {
            return response()->json([
                'alertMessage' => 'You are not Authorized'
            ], 403);
        }

        // Safety check
        if ($role->slug === 'super-admin') {
            return response()->json([
                'message' => 'Super admin role cannot be deleted'
            ], 403);
        }

        try {
            // Detach relations if exist
            if (method_exists($role, 'permissions')) {
                $role->permissions()->detach();
            }

            if (method_exists($role, 'users')) {
                $role->users()->detach();
            }

            $role->delete();

            return response()->json([
                'successMessage' => 'Role deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Helper: convert string to slug
     */
    private function convertToSlug(string $str): string
    {
        $str = strtolower($str);
        return preg_replace('/\s+|[^\w]/', '-', $str);
    }
}
